#!/usr/bin/env python
#! -*- coding: utf-8 -*-
import os
import sys
sys.path.insert(1, os.path.join(sys.path[0], '../..'))

import json
from datetime import date

from collections import defaultdict
from nile.api.v1 import clusters, filters as nf, aggregators as na, extractors as ne, statface as ns
from nile.files import LocalFile
from qb2.api.v1 import extractors as qe
import yt.wrapper as yt

from common import (
    ADNETWORK_POSTBACKS_LOG_PATH,
    JOB_ROOT,
    LAUNCHER_SOFTID,
    MAGIC_DATE_RANGE_STRING_APPREC,
    USERNAME,
    date_string_from_ts,
    get_stat_client)

output_table = '$job_root/apprec_money_by_clid'
yt_token = os.environ['YT_TOKEN']
yt.config["proxy"]["url"] = "hahn"


def load_clids():
    table_path = '//statbox/statbox-dict-by-name/distr_report.json/{}'.format(date.today().isoformat())

    if not yt.exists(table_path):
        raise AttributeError('Can not load clids. Path {} does not exist.'.format(table_path))
    return json.loads(yt.read_file(table_path).read())


def clid1006_to_clid1(clid_dict, softid=LAUNCHER_SOFTID):
    tmp_dict = defaultdict(dict)
    for i in clid_dict:
        if i.get('set.soft_id') != softid:
            continue
        tmp_dict[i['set.caption']][i['clid_type_id']] = i['id']
    result_dict = {int(i['1006']): int(i.get('1')) for i in tmp_dict.values() if type(i) == dict and i.get('1006')}
    result_dict[''] = 'Empty'
    return result_dict


launcher_clid_dict = None


def get_clid1006_to_clid1(clid1006, launcher_clid_dict=launcher_clid_dict):
    try:
        clid1006 = int(clid1006)
    except:
        clid1006 = 0
    return launcher_clid_dict.get(clid1006, 0)


def get_country(country):
    return country or 'undefined'


def main():
    clids_dict = load_clids()
    global launcher_clid_dict
    launcher_clid_dict = clid1006_to_clid1(clids_dict)
    common_path = os.path.join(sys.path[1], 'common.py')
    cluster = clusters \
        .Hahn(pool='{}'.format(USERNAME), token=yt_token) \
        .env(
            templates=dict(job_root=JOB_ROOT, date=MAGIC_DATE_RANGE_STRING_APPREC),
            packages=[LocalFile(common_path)]
        )
    job = cluster.job()
    log = job.table(ADNETWORK_POSTBACKS_LOG_PATH)

    records = log \
        .filter(nf.not_(nf.equals('clid', '-2'))) \
        .project(
            'payout',
            country=ne.custom(get_country, 'country'),
            clid=ne.custom(get_clid1006_to_clid1, 'clid'),
            fielddate=ne.custom(date_string_from_ts, 'timestamp')) \
        .project(
            'payout', 'fielddate',
            qe.unfold_with_total('region', 'country', total='_total_'),
            qe.unfold_with_total('clid1', 'clid', total='_total_')) \
        .groupby('fielddate', 'clid1', 'region') \
        .aggregate(payout_daily=na.sum('payout'),
                   postbacks_count=na.count()
                  ) \
        .put(output_table)

    job.run()

    report = (
        ns.StatfaceReport()
        .path('Mobile_Soft_Launcher/apprec_postbacks')
        .title('AppRec Postbacks')
        .scale('daily')
        .dimensions(
            ns.Date('fielddate').replaceable(),
            ns.StringSelector('clid1').replaceable(),
            ns.StringSelector('region').replaceable(),
        )
        .measures(
            ns.Float('payout_daily').title('payout'),
            ns.Integer('postbacks_count').title('postbacks_count')
        )
    )
    client = get_stat_client()
    (
        report.client(client)
        .data(records.read())
        .publish()
    )


if __name__ == '__main__':
    main()
