#!/usr/bin/env python
# -*- coding: UTF-8 -*-

import os
import sys

from nile.api.v1 import clusters, filters as nf, aggregators as na, extractors as ne, statface as ns
from nile.files import LocalFile
from qb2.api.v1 import filters as qf, extractors as qe

sys.path.insert(1, os.path.join(sys.path[0], '..'))

from common import (
    APPMETRIKA_LOG_PATH,
    DATES_LIST,
    JOB_ROOT,
    LAUNCHER_APIKEY,
    MAGIC_DATE_RANGE_STRING,
    USERNAME,
    get_clid1,
    validate_unicode,
    get_stat_client)

event_types = {'EVENT_CLIENT', 'EVENT_STATBOX'}
yt_token = os.environ['YT_TOKEN']
output_table = '$job_root/funnel'
funnel_event_names = {
    'first_app_launch',
    'intro',
    'settings',
    'make_default',
}
common_path = os.path.join(sys.path[1], 'common.py')
cluster = clusters \
    .Hahn(pool='{}'.format(USERNAME), token=yt_token) \
    .env(
        templates=dict(job_root=JOB_ROOT, date=MAGIC_DATE_RANGE_STRING),
        packages=[LocalFile(common_path)]
    )

job = cluster.job()
log = job.table(APPMETRIKA_LOG_PATH)


def parse_funnel_layer(name, value):
    if not isinstance(value, dict):
        return
    if name == 'first_app_launch':
        return 'first_app_launch'
    elif name == 'intro' and value.get('wallpaper_chose'):
        return 'wallpaper_chose'
    elif name == 'settings':
        if all([
            (value.get('permissions_opened', {}).get('contacts') == {"off":"on"}),
            (value.get('permissions_opened', {}).get('location') == {"off":"on"}),
            (value.get('permissions_opened', {}).get('method') == 'intro'),
            (value.get('permissions_opened', {}).get('sms_email') == {"off":"on"}),
        ]):
            return 'permissions given: all'
        elif all([
            (value.get('permissions_opened', {}).get('contacts') == "off"),
            (value.get('permissions_opened', {}).get('location') == "off"),
            (value.get('permissions_opened', {}).get('method') == 'intro'),
            (value.get('permissions_opened', {}).get('sms_email') == "off"),
        ]):
            return 'permissions given: none'
        elif value.get('permissions_opened', {}).get('method') == 'intro':
            return 'permissions given: some'
    elif name == 'make_default':
        if value.get('whats_next') == {"lnchr_once": ""}:
            return 'make_default: once'
        if value.get('whats_next') == {"lnchr_always": ""}:
            return 'make_default: always'


records = log.qb2(
    log='metrika-mobile-log',
    fields=['device_id', 'event_date', 'event_name',
        qe.mobile.event_value(name='event_value'),
        qe.log_field('Clids_Names', default="[]").rename('clids_names'),
        qe.log_field('Clids_Values', default="[]").rename('clids_values'),
        qe.log_field('Manufacturer', default='unknown'),
        qe.log_field('Model', default='unknown'),
    ],
    filters=[
        qf.region_belongs(master_regions=[225], dictionary='Geobase', field='geo_id'),
        qf.equals('api_key_str', LAUNCHER_APIKEY),
        qf.one_of('event_type', event_types),
        qf.one_of('event_name', funnel_event_names)
    ]) \
    .project(
        'device_id',
        funnel_layer = ne.custom(parse_funnel_layer, 'event_name', 'event_value'),
        Manufacturer=ne.custom(validate_unicode, 'Manufacturer'),
        Model=ne.custom(validate_unicode, 'Model'),
        fielddate='event_date',
        clid=ne.custom(get_clid1, 'clids_names', 'clids_values')
    ) \
    .filter(
        qf.defined('funnel_layer')
    ) \
    .project(
        'device_id',
        'fielddate',
        qe.unfold_with_total('clid1', 'clid', total='_total_'),
        qe.unfold_with_total('layer', 'funnel_layer', total='_total_'),
        qe.unfold_with_total('manufacturer', 'Manufacturer', total='_total_'),
        qe.unfold_with_total('model', 'Model', total='_total_'),
    ) \
    .filter(
        nf.custom(lambda x: x in DATES_LIST, 'fielddate')
    ) \
    .groupby('fielddate', 'clid1', 'layer', 'manufacturer', 'model') \
    .aggregate(
        devices=na.count_distinct('device_id')) \
    .sort('fielddate', 'devices') \
    .put(output_table)
job.run()


report = (
    ns.StatfaceReport()
    .path('Mobile_Soft_Launcher/funnel')
    .title('funnel')
    .scale('daily')
    .dimensions(
        ns.Date('fielddate').replaceable(),
        ns.StringSelector('clid1').replaceable(),
        ns.StringSelector('layer').replaceable(),
        ns.StringSelector('manufacturer').replaceable(),
        ns.StringSelector('model').replaceable()
    )
    .measures(
        ns.Integer('devices').title('devices')
    )
)

client = get_stat_client()
(
    report.client(client)
    .data(records.read())
    .publish()
)
