#!/usr/bin/env python
# -*- coding: utf-8 -*-

import json
import os
from datetime import date, timedelta

import requests
from nile.api.v1 import statface as ns
from nile.api.v1.datetime import date_range

from common import get_stat_client

WEEKS_RECALC_BACK = 2
DATE_END = date.today() - timedelta(days=1)
DATE_START = DATE_END - timedelta(weeks=WEEKS_RECALC_BACK)
DATES_LIST = list(date_range(DATE_START, DATE_END, step=1, stringify=True))
SEARCHES_URL = "https://distribution.yandex.net/api/v2/statistics/basic/report/?lang=ru&oauth_token={}"
OAUTH_TOKEN = os.environ['DISTRIBUTION_TOKEN']


def request_report(params):
    url = SEARCHES_URL.format(OAUTH_TOKEN)
    params = json.dumps(params)
    response = requests.post(url, data=params)
    # response.raise_for_status()
    return response.json()


def fetch(fielddate):
    params = {
        'dimensions': ['clid'],
        'filters': [
            'AND',
                [['soft_id', '=', '1029'],
                ['clid_type_id', '=', '1003']]
        ],
        'measures': ['turnover_all', 'turnover_direct', 'searches'],
        'period': [fielddate, fielddate],
        'currency_convertation': '1'
    }
    return request_report(params)


def process(fielddate):
    searches = 0.0
    revenue = 0.0
    report = fetch(fielddate)['data']['report']
    for i in report:
        for measure in i['measures']:
            if measure['name'] == 'turnover_direct':
                revenue += measure.get('value') or 0
            if measure['name'] == 'searches':
                searches += measure.get('value') or 0
    data = [{
        'fielddate': fielddate,
        'revenue': revenue,
        'searches': searches,
    }]
    return data


def publish(data):
    report = (
        ns.StatfaceReport()
        .path('Mobile_Soft_Launcher/distribution_searches')
        .title('distribution_searches')
        .scale('daily')
        .dimensions(
            ns.Date('fielddate').replaceable()
        )
        .measures(
            ns.Float('revenue').title('revenue (USD)'),
            ns.Float('searches').title('searches')
        )
    )
    client = get_stat_client()
    (
        report.client(client)
        .data(data)
        .publish(smart_update_config=True, upload_config=False)
    )


def main():
    for d in DATES_LIST:
        data = process(d)
        publish(data)


if __name__ == '__main__':
    main()
