#!/usr/bin/env python
# -*- coding: UTF-8 -*-
"""
Takes Launcher Shtorka events from metrika_mobile_log. Draws stat report.
"""
import os
import sys
sys.path.insert(1, os.path.join(sys.path[0], '../..'))

from nile.api.v1 import clusters, aggregators as na, extractors as ne, statface as ns
from nile.files import LocalFile
from qb2.api.v1 import filters as qf, extractors as qe
from common import (
    APPMETRIKA_LOG_PATH,
    DATES_LIST,
    JOB_ROOT,
    LAUNCHER_APIKEY,
    MAGIC_DATE_RANGE_STRING,
    USERNAME,
    get_clid1,
    get_stat_client)

eventname = 'shtorka_opened'
output_table = '$job_root/shtorka'
yt_token = os.environ['YT_TOKEN']

common_path = os.path.join(sys.path[1], 'common.py')
cluster = clusters \
    .Hahn(pool='{}'.format(USERNAME), token=yt_token) \
    .env(
        templates=dict(job_root=JOB_ROOT, date=MAGIC_DATE_RANGE_STRING),
        packages=[LocalFile(common_path)]
    )
job = cluster.job()
log = job.table(APPMETRIKA_LOG_PATH)


def get_layer_1(event_value):
    if not isinstance(event_value.get('whats_next'), dict):
        return 'empty'
    layer_1 = list(event_value['whats_next'].values())[0]
    if isinstance(layer_1, str):
	return layer_1
    try:
        layer_1 = {k: layer_1[k] for k in layer_1 if k not in {'time', 'package_name', 'params'}}
    except Exception:
	    raise Exception(event_value)
    if not layer_1:
        layer_1 = 'empty'
    return str(layer_1)


def get_layer_0(event_value):
    whats_next = event_value.get('whats_next') or 'empty'
    if not isinstance(whats_next, dict):
        return str(whats_next)
    layer_0 = whats_next.keys()[0]
    if not layer_0:
        layer_0 = 'empty'
    return str(layer_0)


def get_method(event_value):
    method = 'empty'
    try:
        params = event_value.get('params')
        method = str(params.get('method', 'empty') or 'empty')
    except:
        pass
    return method

schema = dict(
    clid=str,
    devices=int,
    events=int,
    fielddate=str,
    layer_0=str,
    layer_1=str,
    method=str,
    version=str
)

records = log.qb2(
    log='metrika-mobile-log',
    fields=[
        'app_version', 'device_id', 'event_date',
        qe.mobile.event_value(name='event_value'),
        qe.log_field('Clids_Names', default="[]").rename('clids_names'),
        qe.log_field('Clids_Values', default="[]").rename('clids_values')
    ],
    filters=[
        qf.region_belongs(master_regions=[225], dictionary='Geobase', field='geo_id'),
        qf.equals('api_key_str', LAUNCHER_APIKEY),
        qf.equals('event_name', eventname),
        qf.one_of('event_date', list(DATES_LIST))
    ]) \
    .project(
        'app_version', 'device_id',
        qe.custom('method_', get_method, 'event_value'),
        qe.custom('layer_0_', get_layer_0, 'event_value'),
        qe.custom('layer_1_', get_layer_1, 'event_value'),
        clid_=ne.custom(get_clid1, 'clids_names', 'clids_values'),
        fielddate='event_date',
    ) \
    .filter(
        qf.one_of('method_', {'click', 'swipe', 'intent', 'empty'})        
    ) \
    .project(
        'device_id',
        'fielddate',
        qe.unfold_with_total('clid', 'clid_', total='_total_'),
        qe.unfold_with_total('layer_0', 'layer_0_', total='_total_'),
        qe.unfold_with_total('layer_1', 'layer_1_', total='_total_'),
        qe.unfold_with_total('method', 'method_', total='_total_'),
        qe.unfold_with_total('version', 'app_version', total='_total_'),
    ) \
    .groupby('fielddate', 
        'clid',
        'layer_0',
        'layer_1',
        'version',
        'method') \
    .aggregate(
        events=na.count(),
        devices=na.count_distinct('device_id', in_memory=False)) \
    .sort('fielddate', 'devices') \
    .put(output_table, schema=schema)
job.run()

report = (
    ns.StatfaceReport()
    .path('Mobile_Soft_Launcher/events_shtorka')
    .title('shtorka')
    .scale('daily')
    .dimensions(
        ns.StringSelector('clid').replaceable(),
        ns.Date('fielddate').replaceable(),
        ns.StringSelector('layer_0').replaceable(),
        ns.StringSelector('layer_1').replaceable(),
        ns.StringSelector('method').replaceable(),
        ns.StringSelector('version').replaceable(),
    )
    .measures(
        ns.Integer('devices').title('devices'),
        ns.Integer('events').title('events')
    )
)

client = get_stat_client()
(
    report.client(client)
    .data(records.read())
    .publish()
)
