#!/usr/bin/env python
# -*- coding: UTF-8 -*-
"""
Takes Launcher Shtorka search provider from metrika_mobile_log. Draws stat report.
"""
import os
import sys
sys.path.insert(1, os.path.join(sys.path[0], '../..'))

from nile.api.v1 import clusters, aggregators as na, extractors as ne, statface as ns
from nile.files import LocalFile
from qb2.api.v1 import filters as qf, extractors as qe
from common import (
    APPMETRIKA_LOG_PATH,
    DATES_LIST,
    JOB_ROOT,
    LAUNCHER_APIKEY,
    MAGIC_DATE_RANGE_STRING,
    USERNAME,
    get_clid1,
    get_stat_client)


eventname = 'shtorka_opened'
output_table = '$job_root/shtorka_search_provider'
yt_token = os.environ['YT_TOKEN']

common_path = os.path.join(sys.path[1], 'common.py')
cluster = clusters \
    .Hahn(pool='{}'.format(USERNAME), token=yt_token) \
    .env(
        templates=dict(job_root=JOB_ROOT, date=MAGIC_DATE_RANGE_STRING),
        packages=[LocalFile(common_path)]
    )
job = cluster.job()
log = job.table(APPMETRIKA_LOG_PATH)


def is_search(event_value):
    try:
        result = int(
            event_value['whats_next']['search']['type'] == 'query' or
            event_value['whats_next']['search']['press_enter'] == 'search_query' or
            event_value['whats_next']['search']['click_suggest'] == 'search_query' or
            event_value['whats_next'] == 'search_suggest_tap'
        )
    except (KeyError, TypeError):
        result = 0
    return result


records = log.qb2(
    log='metrika-mobile-log',
    fields=['app_version', 'device_id', 'event_date',
        qe.mobile.event_value(name='event_value'),
        qe.log_field('Clids_Names', default="[]").rename('clids_names'),
        qe.log_field('Clids_Values', default="[]").rename('clids_values')],
    filters=[
        qf.one_of('event_type', {'EVENT_CLIENT', 'EVENT_STATBOX'}),
        qf.equals('api_key_str', LAUNCHER_APIKEY),
        qf.equals('event_name', eventname),
        qf.one_of('event_date', list(DATES_LIST))
    ]) \
    .project(
        'app_version', 'device_id',
        qe.custom('search_provider_',
            lambda x: x.get('params').get('search_provider'),
            'event_value'),
        qe.custom('is_search_', is_search, 'event_value'),
        clid1=ne.custom(get_clid1, 'clids_names', 'clids_values'),
        fielddate='event_date',
    ) \
    .filter(qf.defined('app_version', 'search_provider_', 'is_search_', 'clid1')) \
    .project(
        'device_id', 'fielddate',
        qe.unfold_with_total('version', 'app_version', total='_total_'),
        qe.unfold_with_total('search_provider', 'search_provider_', total='_total_'),
        qe.unfold_with_total('is_search', 'is_search_', total='_total_'),
        qe.unfold_with_total('clid', 'clid1', total='_total_'),
    ) \
    .groupby('fielddate', 'version', 'search_provider', 'is_search', 'clid') \
    .aggregate(
        events=na.count(),
        devices=na.count_distinct('device_id', in_memory=False)) \
    .sort('fielddate', 'devices') \
    .put(output_table)
job.run()


report = (
    ns.StatfaceReport()
    .path('Mobile_Soft_Launcher/shtorka_search_provider')
    .title('shtorka_search_provider')
    .scale('daily')
    .dimensions(
        ns.Date('fielddate').replaceable(),
        ns.StringSelector('version').replaceable(),
        ns.StringSelector('search_provider').replaceable(),
        ns.StringSelector('is_search').replaceable(),
        ns.StringSelector('clid').replaceable(),
    )
    .measures(
        ns.Integer('devices').title('devices'),
        ns.Integer('events').title('events')
    )
)

client = get_stat_client()
(
    report.client(client)
    .data(records.read())
    .publish()
)
