#!/usr/bin/env python
# -*- coding: UTF-8 -*-

import ast
import os
from datetime import date, timedelta

from nile.api.v1 import clusters, aggregators as na, extractors as ne, statface as ns
from nile.api.v1.datetime import date_range
from qb2.api.v1 import filters as qf, extractors as qe

from common import get_stat_client

APIKEY = '37460'
WEEKS_RECALC_BACK = 4
DATE_END = date.today() - timedelta(days=1)
DATE_START = DATE_END - timedelta(weeks=WEEKS_RECALC_BACK)
DATES_LIST = list(date_range(DATE_START, DATE_END, step=1, stringify=True))
# '{2017-01-01..2017-10-31}'
MAGIC_DATE_RANGE_STRING = '..'.join([DATE_START.isoformat(), DATE_END.isoformat()]).join(['{', '}'])
JOB_ROOT = 'home/advisor/chikachoff/events'
OUTPUT_TABLE = '$job_root/weather'
RAW_LOG_TABLE = '//logs/yandex-phone-metrika-mobile-log/1d/@date'
USERNAME = 'chikachoff'
YT_TOKEN = os.environ['YT_TOKEN']


def get_clid1(a, b):
    try:
        a, b = ast.literal_eval(a), ast.literal_eval(b)
        return dict(zip(a, b)).get('clid1') or 'Empty'
    except ValueError:
        return 'Empty'


CLUSTER = clusters \
    .Hahn(pool='search-research_{}'.format(USERNAME), token=YT_TOKEN) \
    .env(templates=dict(job_root=JOB_ROOT, date=MAGIC_DATE_RANGE_STRING))
JOB = CLUSTER.job()
LOG = JOB.table(RAW_LOG_TABLE)


RECORDS = LOG.qb2(
    log='metrika-mobile-log',
    fields=['device_id', 'event_date',
        qe.mobile.event_value(name='event_value'),
        qe.log_field('Clids_Names', default="[]").rename('clids_names'),
        qe.log_field('Clids_Values', default="[]").rename('clids_values')
    ],
    filters=[
        qf.equals('api_key_str', APIKEY),
        qf.one_of('event_date', list(DATES_LIST)),
        qf.equals('event_name', 'weather')
    ]) \
    .project('device_id',
             qe.custom('action_', lambda x: str(x.get('main_screen')) or 'other', 'event_value'),
             fielddate='event_date',
             clid1=ne.custom(get_clid1, 'clids_names', 'clids_values')
    ) \
    .project(
        'device_id', 'fielddate',
        qe.unfold_with_total('action', 'action_', total='_total_'),
        qe.unfold_with_total('clid', 'clid1', total='_total_')
    ) \
    .groupby('fielddate', 'clid', 'action') \
    .aggregate(
        events=na.count(),
        devices=na.count_distinct('device_id', in_memory=False)) \
    .sort('fielddate', 'devices') \
    .put(OUTPUT_TABLE)
JOB.run()

# задаём конфигурацию отчёта на Статфейсе
REPORT = (
    ns.StatfaceReport()
    # задаем путь отчета, тайтл и детализацию по дням
    .path('Mobile_Soft_Launcher/weather')
    .title('weather')
    .scale('daily')
    # описываем измерения отчета
    .dimensions(
        ns.Date('fielddate').replaceable(),
        ns.StringSelector('action').replaceable(),
        ns.StringSelector('clid').replaceable()
    )
    # описываем метрики отчета
    .measures(
        # задаем:
        # - типы отображения полей (view types в Статфейсе)
        # - тайтлы полей
        # - параметры видимости (показывать ли метрику по умолчанию или нет)
        ns.Integer('devices').title('devices'),
        ns.Integer('events').title('events')
    )
)
# создаем клиент для публикации
client = get_stat_client()
(
    # связываем клиент с отчетом
    REPORT.client(client)
    # выгружаем результаты расчета и связываем их с отчетом
    .data(RECORDS.read())
    # публикуем данные на Статфейс
    .publish()
)

