#!/usr/bin/env python
# -*- coding: UTF-8 -*-
import os
import sys
sys.path.insert(1, os.path.join(sys.path[0], '..'))

from nile.api.v1 import (
    clusters,
    aggregators as na,
    statface as ns,
    files as nfl
)
from nile.files import LocalFile
from qb2.api.v1 import filters as qf, extractors as qe, resources as qr
from common import (
    APPMETRIKA_LOG_PATH,
    DATES_LIST,
    JOB_ROOT,
    LAUNCHER_APIKEY,
    MAGIC_DATE_RANGE_STRING,
    USERNAME,
    get_app_ids_from_dict,
    get_stat_client)


output_table = '$job_root/yandex_app_installs'
yt_token = os.environ['YT_TOKEN']

common_path = os.path.join(sys.path[1], 'common.py')
cluster = clusters \
    .Hahn(pool='{}'.format(USERNAME), token=yt_token) \
    .env(
        templates=dict(job_root=JOB_ROOT, date=MAGIC_DATE_RANGE_STRING),
        packages=[LocalFile(common_path)]
    )
job = cluster.job()
log = job.table(APPMETRIKA_LOG_PATH)


app_ids_resource = qr.json('apps_projects.json', get_app_ids_from_dict)
project_names_resource = qr.json('project_names.json')


class YandexAppDetector(object):
    def __init__(self):
        self.app_ids = None

    def __call__(self, event_value):
        if self.app_ids is None:
            self.app_ids = app_ids_resource()
        try:
            return (
                event_value['packageName'] in self.app_ids
            )
        except (KeyError, TypeError):
            return False


records = log.qb2(
    files=(nfl.StatboxDict('project_names.json'), nfl.StatboxDict('apps_projects.json')),
    log='metrika-mobile-log',
    fields=['device_id', 'event_date', 'event_name', qe.mobile.event_value(name='event_value')],
    filters=[
        qf.default_filtering('metrika-mobile-log'),
        qf.equals('api_key_str', LAUNCHER_APIKEY),
        qf.equals('event_name', 'App_install'),
        qf.one_of('event_date', list(DATES_LIST)),
        qf.custom(YandexAppDetector(), 'event_value')
    ]) \
    .project('device_id', fielddate='event_date') \
    .groupby('fielddate') \
    .aggregate(
        events=na.count(),
        devices=na.count_distinct('device_id', in_memory=False)) \
    .sort('fielddate', 'devices') \
    .put(output_table)
job.run()

report = (
    ns.StatfaceReport()
    .path('Mobile_Soft_Launcher/yandex_app_installs')
    .title('yandex_app_installs')
    .scale('daily')
    .dimensions(
        ns.Date('fielddate').replaceable()
    )
    .measures(
        ns.Integer('devices').title('devices'),
        ns.Integer('events').title('events')
    )
)

client = get_stat_client()
(
    report.client(client)
    .data(records.read())
    .publish(smart_update_config=True, upload_config=False)
)
