#!/usr/bin/env python
# -*- coding: UTF-8 -*-

import ast
import os
import re
from datetime import date, datetime

from nile.api.v1 import (
    clusters,
    extractors as ne,
)
from qb2.api.v1 import filters as qf, extractors as qe
from qb2.api.v1.typing import (
    Dict,
    Float,
    UInt64,
    Optional,
    String,
)

config = {
    'manufacturer': 'Yandex',
    'model': 'YNDX-000SB',
    'eventtypes': {'EVENT_FIRST', 'EVENT_INIT', 'EVENT_CLIENT', 'EVENT_STATBOX'},
    'job_root': os.environ.get('JOB_ROOT', 'home/advisor/agregat'),
    'search_url_pattern': re.compile('^https://yandex.*/search'),
    'username': '',
    'yt_token': os.environ.get('YT_TOKEN'),
    'yt_input_table': os.environ.get('YT_INPUT_TABLE'),
    'yt_output_table': os.path.join('$job_root', os.environ.get('YT_INPUT_TABLE').split('/')[-1]),
}

cluster = clusters \
    .Hahn(
        pool='{}'.format(config['username']),
        token=config['yt_token']
    ) \
    .env(
        templates=dict(
            job_root=config['job_root'],
        ),
        yt_spec_defaults=dict(
            pool_trees=["physical"],
            tentative_pool_trees=["cloud"]
        )
    )
job = cluster.job()
log = job.table(config['yt_input_table'])


def get_clid(a, b, clid_type):
    try:
        a, b = ast.literal_eval(a), ast.literal_eval(b)
        return int(dict(zip(a, b)).get(clid_type))
    except (ValueError, TypeError,):
        return None


def get_method(event_value):
    try:
        return event_value.get('params', {}).get('method')
    except Exception:
        return None


def get_place(event_value):
    if event_value:
        placement = event_value.get('place') or event_value.get('placement')
        return placement
    return None


def get_package_name(event_value):
    if event_value:
        package_name = event_value.get('packageName') or event_value.get('package_name')
        return package_name
    return None


def detect_yabro_search(app_project, search_url_pattern, event_name, event_value):
    if 'Mobile_Soft_Browser' != app_project:
        return None
    if not isinstance(event_value, dict):
        return None
    bitch_detected = 'is_yandex_browser_search'
    url = event_value.get('url')
    if 'url opened' != event_name:
        return None
    source = event_value.get('source')
    if not (source and url):
        return None
    if re.match(search_url_pattern, url):
        if 'refresh' in source:
            return bitch_detected


def detect_launcher_search(app_project, event_name, event_value):
    if 'Mobile_Soft_Launcher' != app_project:
        return None
    if 'shtorka_opened' != event_name:
        return None
    whats_next = event_value.get('whats_next')
    if not whats_next:
        return None
    # shtorka layer0 suggest
    if 'search_suggest_tap' == whats_next:
        return 'suggest_layer_0'
    # shtorka layer1: suggest and search
    if not isinstance(whats_next, dict):
        return None
    if whats_next.get('search', {}).get('type') == 'query':
        return whats_next.get('search', {}).get('action')
    return None


def detect_searchapp_search(app_project, event_name):
    if 'Search_Mobile_App' != app_project:
        return None
    if 'CLIENT_SEARCH_REQUEST_EVENT' == event_name:
        return 'is_search_app_search'


def detect_browser_launch(app_project, event_name, event_value):
    if 'Mobile_Soft_Launcher' != app_project:
        return None
    if 'app_launch' != event_name:
        return None
    if 'com.android.chrome' == event_value.get('packageName'):
        return 'is_chrome_browser_launch'
    if 'com.yandex.browser' == event_value.get('packageName'):
        return 'is_yandex_browser_launch'
    if 'ru.yandex.searchplugin' == event_value.get('packageName'):
        return 'is_yandex_search_app_launch'


def detect_chrome_widget_launch(app_project, event_name, event_value):
    if 'Mobile_Soft_Launcher' != app_project:
        return None
    if 'widget_launch' != event_name:
        return None
    if 'com.android.chrome' == event_value.get('packageName'):
        return 'is_chrome_widget_launch'


def get_target(app_project, event_name, event_value):
    if detect_yabro_search(app_project, config['search_url_pattern'], event_name, event_value):
        return detect_yabro_search(app_project, config['search_url_pattern'], event_name, event_value)
    if detect_searchapp_search(app_project, event_name):
        return detect_searchapp_search(app_project, event_name)
    if detect_launcher_search(app_project, event_name, event_value):
        return detect_launcher_search(app_project, event_name, event_value)
    if detect_browser_launch(app_project, event_name, event_value):
        return detect_browser_launch(app_project, event_name, event_value)
    if detect_chrome_widget_launch(app_project, event_name, event_value):
        return detect_chrome_widget_launch(app_project, event_name, event_value)

def ts_to_date(ts):
    return date.fromtimestamp(ts).isoformat()


def ts_to_datetime(ts):
    return datetime.fromtimestamp(ts).isoformat()


SCHEMA = dict(
    api_key_str=str,
    app_project=Optional[String],
    clid1=Optional[UInt64],
    clid1003=Optional[UInt64],
    clid1006=Optional[UInt64],
    clid1010=Optional[UInt64],
    app_version=str,
    device_id=str,
    event_date=str,
    event_datetime=str,
    event_name=str,
    event_number=UInt64,
    event_timestamp=UInt64,
    event_type=str,
    event_value=Optional[Dict[String, String]],
    geo_id=UInt64,
    latitude=Optional[Float],
    longitude=Optional[Float],
    method=Optional[String],
    model=str,
    operator_id=Optional[UInt64],
    place=Optional[Dict[String, String]],
    package_name=Optional[Dict[String, String]],
    session_id=str,
    session_type=str,
    target=Optional[String],
    uuid=str
)


def run():
    stream = log.qb2(
        log='metrika-mobile-log',
        fields=[
            'api_key_str',
            'app_project',
            'app_version',
            'device_id',
            'event_name',
            'event_number',
            'event_type',
            'geo_id',
            'latitude',
            'longitude',
            'operator_id',
            'session_id',
            'session_type',
            'uuid',
            qe.const('clid1', 'clid1'),
            qe.const('clid1003', 'clid1003'),
            qe.const('clid1006', 'clid1006'),
            qe.const('clid1010', 'clid1010'),
            qe.integer_log_field('EventTimestamp').rename('event_timestamp').allow_override(),
            qe.custom('event_date', ts_to_date, 'event_timestamp').allow_override(),
            qe.custom('event_datetime', ts_to_datetime, 'event_timestamp').allow_override(),
            qe.mobile.event_value(name='event_value'),
            qe.log_field('Manufacturer').rename('manufacturer'),
            qe.log_field('Model').rename('model'),
            qe.log_field('Clids_Names', default="[]").rename('clids_names'),
            qe.log_field('Clids_Values', default="[]").rename('clids_values'),
        ],
        filters=[
            qf.equals('manufacturer', config['manufacturer']),
            qf.equals('model', config['model']),
            qf.one_of('event_type', config['eventtypes']),
        ]) \
        .project(
            ne.all(exclude=('clids_names', 'clids_values', 'manufacturer')),
            clid1=ne.custom(get_clid, 'clids_names', 'clids_values', 'clid1'),
            clid1003=ne.custom(get_clid, 'clids_names', 'clids_values', 'clid1003'),
            clid1006=ne.custom(get_clid, 'clids_names', 'clids_values', 'clid1006'),
            clid1010=ne.custom(get_clid, 'clids_names', 'clids_values', 'clid1010'),
            method=ne.custom(get_method, 'event_value'),
            place=ne.custom(get_place, 'event_value'),
            package_name=ne.custom(get_package_name, 'event_value'),
            target=ne.custom(get_target, 'app_project', 'event_name', 'event_value'),
        )

    stream.put(config['yt_output_table'], schema=SCHEMA)

    job.run()


if __name__ == '__main__':
    run()
