import os
from sys import platform

import argparse
from vault_client.instances import Production as VaultClient

SECRETS_TAG = 'launcher_unified'

if platform == "linux" or platform == "linux2":
    BASHRC_NAME = '~/.bashrc'
elif platform == "darwin":
    BASHRC_NAME = '~/.bash_profile'
else:
    print('Unsupported platform: %s' % platform)
    sys.exit(1)

BASHRC_PATH = os.path.expanduser(BASHRC_NAME)

parser = argparse.ArgumentParser(description='Download launcher secrets from yav.yandex-team.ru')
parser.add_argument('-o', action='store', type=str, dest='output',
                    help='File to save secrets to. By default outputs to stdout without "export"')
parser.add_argument('--bashrc', '-b', action='store_true', dest='bashrc',
                    help='Add source line to bashrc')
parser.add_argument('--nochmode', '-c', action='store_true', dest='nochmode',
                    help='Do not chmod 600 on output file')

if __name__ == '__main__':
    args = parser.parse_args()
    if not args.output and (args.bashrc or args.nochmode):
        raise parser.error('Can not use -b or -c options without -o')
    client = VaultClient(decode_files=True)
    secrets = client.list_secrets(tags=[SECRETS_TAG])

    env_dict = {}
    for secret in secrets:
        version_name = secret['last_secret_version']['version']
        version = client.get_version(version_name)
        env_dict.update(version['value'])

    export_string = 'export ' if args.output else ''
    secrets_string = ''.join('{}{}="{}"\n'.format(export_string, key, value)
                             for key, value in sorted(env_dict.items()))

    if args.output:
        with open(args.output, 'w') as secrets_file:
            secrets_file.write(secrets_string)

        if not args.nochmode:
            os.chmod(args.output, 0o600)

        if args.bashrc:
            bashrc_string = '\nsource {}\n'.format(os.path.realpath(args.output))
            with open(BASHRC_PATH, 'r') as bashrc_file:
                bashrc_has_launcher = args.output in bashrc_file.read()
            if not bashrc_has_launcher:
                with open(BASHRC_PATH, 'a') as bashrc_file:
                    bashrc_file.write(bashrc_string)
    else:
        print(secrets_string)
