import json
import logging

from django.conf import settings
from updater.consts import Keys, CLIENT_PROPERTIES_PREFIX

logger = logging.getLogger(__name__)


def is_client_property(item):
    return item.name.startswith(CLIENT_PROPERTIES_PREFIX)


_necessary_keys = ('beta_name', 'package_name', 'branch')


def _parse_tracked_builds(raw_value):
    if raw_value is None:
        return None
    try:
        value = json.loads(raw_value)
    except ValueError:
        logger.warning('%s in not a valid json', raw_value)
        return None

    if not isinstance(value, list):
        return None

    for item in value:
        if not isinstance(item, dict):
            return None

        if not all((key in item for key in _necessary_keys)):
            return None

    return value


def _get_tracked_builds(config, client_prefix):
    def make_vc_conditions_case_insensitive():
        for build in builds:
            build[Keys.VISIBILITY_CONDITIONS] = [key.lower() for key in build.get(Keys.VISIBILITY_CONDITIONS, [])]
        return builds

    value = config.get_value(client_prefix + Keys.TRACKED_BUILDS)
    if value is None:
        logger.warning('client_prefix="%s" is applied but appropriate config not found', client_prefix, extra={'sample_rate': 0.1})
        value = config.get_value(Keys.TRACKED_BUILDS)

    builds = _parse_tracked_builds(value)
    if builds is not None:
        return make_vc_conditions_case_insensitive()

    logger.warning('use default value for localization key=%s', Keys.TRACKED_BUILDS)
    return settings.TRACKED_BUILDS


def get_tracked_builds(config, client_prefix, client_tags):
    builds = _get_tracked_builds(config, client_prefix)
    if not client_tags:  # None or [] - False both
        return filter(lambda x: not bool(x.get(Keys.VISIBILITY_CONDITIONS)), builds)

    def filter_func(build):
        vc = set(build.get(Keys.VISIBILITY_CONDITIONS, []))
        return client_tags.issubset(vc)

    return filter(filter_func, builds)
