import re
import json
import logging
from uuid import UUID
from django.conf import settings
from rest_framework.exceptions import APIException
from rest_framework import status
from localization import LocalizationUser, UserSpecificConfig, Translator
from updater.consts import Keys

logger = logging.getLogger(__name__)


class BadRequestAPIError(APIException):
    status_code = status.HTTP_400_BAD_REQUEST


class HttpClientMixin(object):
    def initial(self, request):
        super(HttpClientMixin, self).initial(request)
        self.headers = request.META

    def get_header(self, name, default=None):
        return self.headers.get('HTTP_%s' % name, default)

    @property
    def uuid(self):
        uuid_string = self.get_header('X_YAUUID', '')
        try:
            return UUID(uuid_string)
        except ValueError:
            raise BadRequestAPIError('Could not construct UUID from "%s" string' % uuid_string)

    @property
    def client_tags(self):
        tags = self.get_header('X_UPDR_TAGS')
        if not tags:
            return None

        client_tags = set(tag.strip().lower() for tag in tags.split(','))
        client_tags.discard('')
        return client_tags

    @property
    def user_agent(self):
        return self.get_header('USER_AGENT')

    @property
    def yphone_id(self):
        return self.get_header('YPHONE_ID')

    @property
    def serial_number(self):
        return self.get_header('SERIAL')


class TrackedBuilds(object):
    default_prefix = ''

    def __init__(self, localization_config):
        self.localization_config = localization_config

    @staticmethod
    def _extract_client_id(http_client):
        ua = http_client.get_header('USER_AGENT')
        return ua.partition('/')[0]

    def get_key_prefix(self, http_client):
        client_id = self._extract_client_id(http_client)
        if client_id not in self._clients_mapping:
            return self.default_prefix

        app_mapping = self._clients_mapping[client_id]
        if isinstance(app_mapping, list):
            for item in app_mapping:
                for header, value in item['conditions'].iteritems():
                    header_value = http_client.get_header(header)
                    if (value and header_value is not None) or (not value and header_value is None):
                        return item['value']

            logger.warning('cliend_id=%s has own config section but none of the condition was met', client_id)
            return self.default_prefix
        else:
            return app_mapping['value']

    @property
    def _clients_mapping(self):
        value = self.localization_config.get_value(Keys.CLIENTS_MAPPING, default_value='{}')
        try:
            mapping = json.loads(value)
            if isinstance(mapping, dict):
                return mapping

            raise TypeError
        except (TypeError, ValueError):
            logger.warning('Can\'t parse clients mapping from source: "%s"', value, extra={'sample_rate': 0.05})
            return {}


DEVICE_PHONE = 'phone'
DEVICE_TABLET = 'tablet'
# regexp is almost the same as in advisor
USER_AGENT_REGEXP = re.compile("^(?P<app_name>[\w.]+)/(?P<app_version_string>[\w.]+)?\s+"
                               "\((?P<device_manufacturer>\S+)\s+(?P<device_model>.*);\s+"
                               "(?P<os_name>.+)\s+(?P<os_version>.*)\)\s*"
                               "(?P<device_type>(?i)({}|{}))?".format(DEVICE_PHONE, DEVICE_TABLET))


def create_localization_user(uuid, locale=None, user_agent=None, serial_number=None, yphone_id=None):

    def prepare_device_type(ua_device_type):
        return ua_device_type.lower() if ua_device_type else DEVICE_PHONE

    user = LocalizationUser(uuid=uuid)
    if locale is not None:
        user.language = locale.language
        user.country = locale.country

    if serial_number is not None:
        user.serial_number = serial_number

    if yphone_id is not None:
        user.yphone_id = yphone_id

    if user_agent is None:
        return user

    if user_agent == 'automotive-updater':
        # Yandex.Auto updater uses USER-AGENT=automotive-updater
        # https://wiki.yandex-team.ru/Automotive/serverdevelopment/auto-updater/architectural/
        user.app_name = user_agent
    else:
        user_agent_match = USER_AGENT_REGEXP.match(user_agent)
        if user_agent_match is not None:
            user.app_name = user_agent_match.group('app_name')
            user.app_version = user_agent_match.group('app_version_string')
            user.device_manufacturer = user_agent_match.group('device_manufacturer')
            user.device_model = user_agent_match.group('device_model')
            user.device_type = prepare_device_type(user_agent_match.group('device_type'))
            user.os_name = user_agent_match.group('os_name')
            user.os_version = user_agent_match.group('os_version')

    return user


def get_localization_helpers(user):
    project_name = settings.LOCALIZATION_PROJECT
    config = UserSpecificConfig(user, project_name)
    translator = Translator(user, settings.LOCALIZATION_TRANSLATIONS_PROJECT)

    return config, translator
