import logging


def make_http_header(header):
    return 'HTTP_%s' % header.upper().replace('-', '_').replace('.', '_')


HEADERS_FOR_YAMISC = {
    name: make_http_header(name)
    for name in ('X-Updr-Tags',)
}


class FixEmptyHostMiddleWare(object):
    """
    Used to fix bug with response code 400 when binding to [::] and empty Host header.
    Django tries to use SERVER_NAME as host in django.http.request:HttpRequest.get_host()
    but "::" does not fit django.http.request:host_validation_re regular expression.

    This fix works with Django=1.8.4. Please review it when upgrading django.
    """

    @staticmethod
    def process_request(request):
        if 'HTTP_HOST' not in request.META and request.META.get('SERVER_NAME', '').startswith(':'):
            request.META['SERVER_NAME'] = 'localhost'


class ExceptionLoggingMiddleware(object):
    # noinspection PyUnusedLocal
    def process_exception(self, request, exception):
        logging.exception('Uncaught exception', extra={'request': request})


class HttpHeadersLoggingMiddleware(object):
    """
    this class is used for passing some required for logging headers into 'X-YaMisc' header.
    """

    def process_response(self, request, response):
        selected_headers = {}
        for name, http_header in HEADERS_FOR_YAMISC.items():
            if http_header in request.META:
                selected_headers[name] = request.META[http_header]
        if selected_headers:
            response['X-YaMisc'] = '|'.join('='.join(pair) for pair in selected_headers.items())
        return response
