import logging
from datetime import datetime

import mongoengine as me
from django.conf import settings
# noinspection PyProtectedMember
from mongoengine.connection import _connection_settings
from pymongo import ReadPreference

logger = logging.getLogger(__name__)
DEFAULT_LANGUAGE = 'ru'

try:
    me.connect(
        host=settings.MONGO_URI,
        read_preference=ReadPreference.NEAREST,
        connect=False
    )
except:
    logger.exception('Cant connect to mongo with mongoengine')


class UpdateSubscriptions(me.Document):
    device_id = me.UUIDField(required=True)
    package_name = me.StringField(required=True)
    timestamp = me.DateTimeField(required=False, default=datetime.utcnow)

    meta = {
        'strict': False,
        'indexes': [
            {
                'fields': ('device_id', 'package_name')
            }
        ],
        'index_background': True,
        'auto_create_index': False,
    }


class LangSpecificValues(me.EmbeddedDocument):
    lang = me.StringField()
    title = me.StringField()
    description = me.StringField()


class YandexDistributedApp(me.Document):
    package_name = me.StringField(required=True, unique=True)
    texts = me.ListField(field=me.EmbeddedDocumentField(LangSpecificValues))
    offer_id = me.StringField(required=False)

    def get_title(self, language):
        titles = {item.lang: item.title for item in self.texts}
        return titles.get(language) or titles.get(DEFAULT_LANGUAGE)

    @property
    def is_promo(self):
        return self.offer_id is not None

    meta = {
        'strict': False,
        'indexes': ['package_name'],
        'index_background': True,
    }


def check_mock():
    for alias, connection in _connection_settings.iteritems():
        assert connection.get('is_mock'), "You can load fixtures only in mocked database. " \
                                          "Please check how do you run tests"
