# coding=utf-8
import json
from operator import itemgetter
from unittest import skip

import mock
from django.test import SimpleTestCase
from django.test.client import Client

from fixtures import DatabaseFixturesMixin
from updater.helpers import TrackedBuilds, HttpClientMixin, create_localization_user, get_localization_helpers
from updater.mock.localization import UserSpecificConfig, localization_values
from updater.mock.mysql import AppFile, FirmwareFile

DEFAULT_TEST_UUID = DEFAULT_TEST_DEVICEID = '123456789012345678901234567890ab'

client = Client(HTTP_USER_AGENT='com.yandex.launcher.updater_app/1.0 (LGE Nexus 5; Android 5.1.1)',
                HTTP_X_YAUUID=DEFAULT_TEST_UUID, HTTP_HOST='updater.localhost')

MOST_COMMON_PATH = '/api/v1/updates/?'
MOST_COMMON_PARAMS = {'locale': 'ru_RU'}

TAGGED_TRACKED_BUILDS = [
    {
        'beta_name': 'launcher_updater',
        'package_name': 'com.yandex.launcher.updaterapp',
        'branch': 'dev',
        'visibility_conditions': ['x86', 'classic', 'unpack']
    }
]

NOT_TAGGED_TRACKED_BUILDS = [
    {
        'beta_name': 'launcher_updater',
        'package_name': 'com.yandex.launcher.updaterapp',
        'branch': 'dev',
    },
    {
        "beta_name": "yphone_rom",
        "package_name": "yphone_rom",
        "branch": "dev_test"
    }
]


class HttpClientInfo(object):
    def __init__(self, headers):
        self.mixin = HttpClientMixin()
        self.mixin.headers = headers

        self.get_header = self.mixin.get_header


updater_ua = 'com.yandex.launcher.updaterapp/1.0.0.214 (Fly FS509; Android 6.0)'
updater_no_version_ua = 'com.yandex.launcher.updaterapp/ (Fly FS509; Android 6.0)'
updater_tablet_ua = 'com.yandex.launcher/2.2.5.5002515 (HUAWEI HRY-LX1T; Android 9) Tablet'
browser_ua = 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/34.0.1847.137 Safari/537.36'
automotive_ua = 'automotive-updater'
unknown_client_ua = 'python-requests/2.13.0'


class UpdaterTestBase(SimpleTestCase):
    def setUp(self):
        self.tracked_builds = TrackedBuilds(UserSpecificConfig())

    def valid_response_checks(self, response):
        self.assertEqual(response.status_code, 200)

        parsed_response = json.loads(response.content)
        self.assertGreater(len(parsed_response), 0)

    def check_answer_len(self, response, n):
        self.assertEqual(response.status_code, 200)
        parsed_response = json.loads(response.content)
        self.assertEqual(len(parsed_response), n)

    def check_show(self, response):
        return self.check_answer_len(response, 1)

    def check_not_show(self, response):
        return self.check_answer_len(response, 0)

    def get_response(self, path=MOST_COMMON_PATH, params=MOST_COMMON_PARAMS, tags=None, custom_ua=None,
                     custom_uuid=None):
        kwargs = {'data': params}
        if tags:
            kwargs['HTTP_X_UPDR_TAGS'] = tags
        if custom_ua:
            kwargs['HTTP_USER_AGENT'] = custom_ua
        if custom_uuid:
            kwargs['HTTP_X_YAUUID'] = custom_uuid

        with self.settings(APK_DOWNLOAD_SECRET_KEY=''):
            return client.get(path, **kwargs)


@mock.patch('updater.helpers.Translator', UserSpecificConfig)
@mock.patch('updater.helpers.UserSpecificConfig', UserSpecificConfig)
@mock.patch('updater.views.get_app_file_from_db', lambda *args, **kwargs: AppFile())
@mock.patch('updater.config._get_tracked_builds', lambda *_: TAGGED_TRACKED_BUILDS)
@mock.patch('updater.app_info_loader.app_info_loader.get_apps_info', lambda *args, **kwargs: {})
class UpdaterTaggedBuildsTest(UpdaterTestBase):
    def test_with_valid_tags(self):
        self.check_show(
            self.get_response(tags='x86,classic,unpack')
        )

    def test_tags_mismatch(self):
        self.check_not_show(
            self.get_response(tags='arm,classic,unpack')
        )

    def test_strip_header_value(self):
        self.check_show(
            self.get_response(tags=' x86,  classic, unpack ')
        )

    def test_ignore_case_in_tags(self):
        self.check_show(
            self.get_response(tags='X86,CLASSIC, UNPACK ')
        )

    def test_not_tagged_client_tagged_builds(self):
        self.check_not_show(
            self.get_response()
        )


@mock.patch('updater.helpers.Translator', UserSpecificConfig)
@mock.patch('updater.helpers.UserSpecificConfig', UserSpecificConfig)
@mock.patch('updater.views.get_app_file_from_db', lambda *args, **kwargs: AppFile())
@mock.patch('updater.config._get_tracked_builds', lambda *_: [])
@mock.patch('updater.app_info_loader.app_info_loader.get_apps_info', lambda *args, **kwargs: {})
class UpdaterEmptyBuildsTest(UpdaterTestBase, DatabaseFixturesMixin):
    def setUp(self):
        self.load_fixtures()
        super(UpdaterEmptyBuildsTest, self).setUp()

    def tearDown(self):
        self.cleanup_fixtures()

    def test_market_links(self):
        params = MOST_COMMON_PARAMS.copy()
        params['device_id'] = DEFAULT_TEST_DEVICEID
        params['market_links'] = 1
        response = self.get_response(params=params)

        self.assertEqual(response.status_code, 200)
        answer = json.loads(response.content)
        self.assertGreater(len(answer), 0)

        for item in answer:
            self.assertEqual(item['type'], 'google_play')
            self.assertEqual(item['package_name'], 'com.yandex.launcher')

    def test_google_play(self):
        params = MOST_COMMON_PARAMS.copy()
        params['device_id'] = DEFAULT_TEST_DEVICEID
        params['type'] = ['google_play']
        response = self.get_response(params=params)

        self.assertEqual(response.status_code, 200)
        answer = json.loads(response.content)
        self.assertGreater(len(answer), 0)

        for item in answer:
            self.assertEqual(item['type'], 'google_play')
            self.assertEqual(item['package_name'], 'com.yandex.launcher')

    def test_validation_market_links(self):
        params = MOST_COMMON_PARAMS.copy()
        params['market_links'] = 1
        # use default value for device_id (is None)
        self.assertEqual(self.get_response(params=params).status_code, 400)

    def test_validation_type_google_play(self):
        params = MOST_COMMON_PARAMS.copy()
        params['type'] = ['google_play']
        # use default value for device_id (is None)
        self.assertEqual(self.get_response(params=params).status_code, 400)


@mock.patch('updater.helpers.Translator', UserSpecificConfig)
@mock.patch('updater.helpers.UserSpecificConfig', UserSpecificConfig)
@mock.patch('updater.views.get_app_file_from_db', lambda *args, **kwargs: AppFile())
@mock.patch('updater.config._get_tracked_builds', lambda *_: NOT_TAGGED_TRACKED_BUILDS)
@mock.patch('updater.app_info_loader.app_info_loader.get_apps_info', lambda *args, **kwargs: {})
class UpdaterNotTaggedBuildsTest(UpdaterTestBase):
    def test_usual(self):
        self.valid_response_checks(self.get_response())

    def test_trailing_slash_free(self):
        self.valid_response_checks(self.get_response(path='/api/v1/updates?'))

    def test_locale_param(self):
        for locale in ('', 'ru', 'en_US', 'zh_CN_#Hans', 'de__POSIX'):
            response = self.get_response(params={'locale': locale})
            self.assertEqual(response.status_code, 200)

    def test_uuid_is_required(self):
        response = self.get_response(custom_uuid='bad uuid')
        self.assertEqual(response.status_code, 400)

    @mock.patch('updater.helpers.UserSpecificConfig', UserSpecificConfig)
    def test_settings_view(self):
        self.valid_response_checks(self.get_response(path='/api/v1/settings?', params={}))

    def test_correct_header_treatment(self):
        wrong_header_examples = (
            ',   ',
            ' ,',
            ','
        )

        for item in wrong_header_examples:
            self.check_show(
                self.get_response(tags=item)
            )

    def test_tagged_client_not_tagged_builds(self):
        self.check_not_show(
            self.get_response(tags='arm,classic,unpack')
        )

    def test_not_tagged_client_not_tagged_builds(self):
        self.check_show(
            self.get_response()
        )

    def test_with_valid_uuid(self):
        self.check_show(
            self.get_response(custom_uuid='2516d3cff3ff5de96c2f69d4497063d6')
        )

    def test_with_invalid_uuid(self):
        response = self.get_response(custom_uuid='12345')
        self.assertEqual(response.status_code, 400)

    def test_own_client_config_incorrect_mapping(self):
        headers = {
            'HTTP_USER_AGENT': updater_ua,
            'HTTP_X_UPDR_TAGS': 'auto,ru'
        }

        old_value = localization_values['clients_mapping']
        localization_values['clients_mapping'] = 'value'
        prefix = self.tracked_builds.get_key_prefix(HttpClientInfo(headers))
        self.assertEqual(prefix, '', 'not empty prefix: [%s]' % prefix)
        localization_values['clients_mapping'] = old_value

    def test_own_client_config_correct_mapping(self):
        def check_prefix(ua, etalon_prefix):
            headers = {
                'HTTP_USER_AGENT': ua,
            }

            prefix = self.tracked_builds.get_key_prefix(HttpClientInfo(headers))
            self.assertEqual(prefix, etalon_prefix,
                             'prefix mismatch: actual=[%s], required=[%s]' % (prefix, etalon_prefix))

        ua_to_prefix = {
            updater_ua: 'launcher.',
            browser_ua: 'browser.',
            unknown_client_ua: ''
        }
        for ua, prefix in ua_to_prefix.iteritems():
            check_prefix(ua, prefix)

    def own_service_config_check(self, service, user_agent, tags=None):
        response = self.get_response(custom_ua=user_agent, tags=tags)

        result = map(itemgetter('package_name'), json.loads(response.content))
        for item in result:
            self.assertTrue(service in item, 'not %s config applied, [%s] found' % (service, item))

    def test_own_launcher_config(self):
        self.own_service_config_check('launcher', updater_ua)

    def test_own_auto_config(self):
        self.own_service_config_check('yap', updater_ua, tags='carsharing')

    def test_own_browser_config(self):
        self.own_service_config_check('browser', updater_ua, tags='arm, classic')

    @mock.patch('updater.views.get_app_file_from_db', lambda *args, **kwargs: FirmwareFile())
    def test_firmware_build(self):
        params = MOST_COMMON_PARAMS.copy()
        params['type'] = ['firmware']
        self.check_show(self.get_response(params=params))


@mock.patch('updater.helpers.Translator', UserSpecificConfig)
@mock.patch('updater.helpers.UserSpecificConfig', UserSpecificConfig)
@mock.patch('updater.views.get_app_file_from_db', lambda *args, **kwargs: AppFile())
class UpdaterKeyPrefixTest(UpdaterTestBase):
    @mock.patch('updater.helpers.TrackedBuilds.get_key_prefix', lambda self, client: 'empty_list.')
    def test_empty_list_builds(self):
        response = self.get_response()
        self.assertEqual(json.loads(response.content), [])

    @mock.patch('updater.helpers.TrackedBuilds.get_key_prefix', lambda self, client: 'empty.')
    @mock.patch('updater.config.settings.TRACKED_BUILDS', [{'beta_name': 'a', 'package_name': 'b', 'branch': 'c'}])
    def test_empty_builds(self):
        self.check_answer_len(self.get_response(), n=1)


class UserAgentLocalizationTest(SimpleTestCase):
    def test_updaterapp_good(self):
        user = create_localization_user(uuid=DEFAULT_TEST_UUID, user_agent=updater_ua)
        self.assertEqual(user.app_version, '1.0.0.214')
        self.assertEqual(user.device_manufacturer, 'Fly')
        self.assertEqual(user.device_model, 'FS509')
        self.assertEqual(user.os_name, 'Android')
        self.assertEqual(user.os_version, '6.0')

    def test_updaterapp_no_version(self):
        user = create_localization_user(uuid=DEFAULT_TEST_UUID, user_agent=updater_no_version_ua)
        self.assertEqual(user.app_version, None)
        self.assertEqual(user.device_manufacturer, 'Fly')
        self.assertEqual(user.device_model, 'FS509')
        self.assertEqual(user.os_name, 'Android')
        self.assertEqual(user.os_version, '6.0')

    def test_updaterapp_auto(self):
        user = create_localization_user(uuid=DEFAULT_TEST_UUID, user_agent=automotive_ua)
        self.assertEqual(user.app_name, 'automotive-updater')
        self.assertEqual(user.app_version, None)
        self.assertEqual(user.device_manufacturer, None)
        self.assertEqual(user.device_model, None)
        self.assertEqual(user.os_name, None)
        self.assertEqual(user.os_version, None)

    def test_tablet_ua(self):
        user = create_localization_user(uuid=DEFAULT_TEST_UUID, user_agent=updater_tablet_ua)
        self.assertEqual(user.device_type, 'tablet')

    def test_browser(self):
        user = create_localization_user(uuid=DEFAULT_TEST_UUID, user_agent=browser_ua)
        self.assertEqual(user.app_name, None)
