import hashlib
import logging
import six
from collections import namedtuple
from datetime import datetime
from functools import partial

from django.conf import settings
from django.http.response import HttpResponse, HttpResponseRedirect
from memoize import memoize
from rest_framework import serializers, status
from rest_framework.exceptions import NotFound
from rest_framework.response import Response
from rest_framework.views import APIView

from updater.app_info_loader import app_info_loader, AppInfoLoaderException
from updater.config import get_tracked_builds, is_client_property
from updater.helpers import HttpClientMixin, TrackedBuilds, create_localization_user, get_localization_helpers
from updater.models import AppFile
from updater.mongo import UpdateSubscriptions, YandexDistributedApp
from updater.serializers import GpUpdateSerializer, LocalUpdateSerializer, UpdateTypes
from updater.storage_api import get_download_url, get_local_promo_download_url

SPECIAL_APP_NAME = 'yphone_setup_wizard'

LocalBuildInfo = namedtuple(
    'LocalBuildInfo',
    [
        'package_name',
        'app_name',
        'version_name',
        'version_code',
        'download_url',
        'size',
        'type'
    ]
)

GpBuildInfo = namedtuple(
    'GpBuildInfo',
    [
        'package_name',
        'version_name',
        'version_code',
        'title',
        'type'
    ]
)

logger = logging.getLogger(__name__)


def resolve_package_name(mapping, beta_name):
    for item in mapping:
        if item['beta_name'] == beta_name:
            return item['package_name']


def get_app_file_from_db(app, branch, db='updater_db'):
    return AppFile.objects.using(db).filter(
        stored_to_returning__branch__name=branch,
        stored_to_returning__branch__app__name=app
    ).select_related().first()


def _get_builds_info(app_builds, resolver, translator):
    apps_info = {}
    missing_titles = set()
    for app_info in app_builds:
        package_name = resolver(beta_name=app_info['name'])
        missing_titles.add(package_name)
        apps_info[package_name] = app_info

    try:
        loaded_apps_info = app_info_loader.get_apps_info(
            missing_titles,
            translator.user.language,
            raise_on_missing=False
        )
        for package_name, app_info in six.iteritems(loaded_apps_info):
            if isinstance(app_info['name'], str):
                app_info['name'] = app_info['name'].decode('utf-8')
            apps_info[package_name]['app_title'] = app_info['name']
            missing_titles.remove(package_name)
    except AppInfoLoaderException as exc:
        logger.exception(exc)


    result = []
    for package_name, app_info in six.iteritems(apps_info):
        try:
            app_file = get_app_file_from_db(app_info['name'], app_info['branch'])
            if app_file is None:
                continue
        except Exception as e:
            logger.exception(e)
            continue

        if package_name in missing_titles:
            if app_info['name'] in settings.APP_TO_LOCALIZATION_KEY:
                translation_key = settings.APP_TO_LOCALIZATION_KEY[app_info['name']]
                app_info['app_title'] = translator.translate(translation_key)

            if not app_info.get('app_title'):
                app_info['app_title'] = app_file.stored_to_returning.branch.app.russian1 or app_info['name']

        result.append(LocalBuildInfo(
            package_name=package_name,
            app_name=app_info['app_title'],
            version_name=app_file.stored_to_returning.version,
            version_code=app_file.stored_to_returning.version_code,
            download_url=get_download_url(app_info['name'], app_info['branch'], app_file.filename),
            size=app_file.stored_to_returning.size,
            type=app_info['build_type']
        ))
    return result


def get_beta_updates(tracked_builds, translator, selected_types):
    resolver = partial(resolve_package_name, mapping=tracked_builds)

    app_builds = []
    for item in tracked_builds:
        name, branch = item['beta_name'], item['branch']

        item_is_firmware = bool(name == 'yphone_rom')
        item_type = UpdateTypes.FIRMWARE if item_is_firmware else UpdateTypes.LOCAL

        if item_type not in selected_types:
            continue

        app_builds.append(dict(
            name=name,
            branch=branch,
            build_type=item_type
        ))
    result = _get_builds_info(app_builds, resolver=resolver, translator=translator)

    serializer = LocalUpdateSerializer(result, many=True)
    return serializer.data


class AppNotFound(Exception):
    def __init__(self, package_name):
        super(AppNotFound, self).__init__('package %s not found in updater_db' % package_name)


def get_sw_branch_name(package_name):
    return hashlib.md5(package_name).hexdigest()


def get_sw_app_file(package_name):
    app_file = get_app_file_from_db(SPECIAL_APP_NAME, get_sw_branch_name(package_name), db='setup_wizard_db')
    if app_file is None:
        raise AppNotFound(package_name)

    return app_file


def get_version(package_name):
    app_file = get_sw_app_file(package_name)
    return app_file.stored_to_returning.version, app_file.stored_to_returning.version_code


def get_sw_url(package_name):
    app_file = get_sw_app_file(package_name)
    return get_local_promo_download_url(app_name=SPECIAL_APP_NAME, branch=get_sw_branch_name(package_name),
                                        filename=app_file.filename)


@memoize(timeout=24 * 60 * 60)
def get_title(package_name, language):
    app = YandexDistributedApp.objects.get(package_name=package_name)
    return app.get_title(language)


def get_market_updates(device_id, language):
    result = []
    for item in UpdateSubscriptions.objects(device_id=device_id).distinct('package_name'):
        try:
            title = get_title(item, language)
            name, code = get_version(item)
            result.append(
                GpBuildInfo(
                    package_name=item,
                    version_name=name,
                    version_code=code,
                    title=title,
                    type=UpdateTypes.GOOGLE_PLAY
                )
            )
        except AppNotFound:
            logger.warning('app info not found in updater_db for item=%s', item)
        except YandexDistributedApp.DoesNotExist:
            logger.warning('YandexDistributedApp not found for item=%s', item)

    serializer = GpUpdateSerializer(result, many=True)
    return serializer.data


# noinspection PyUnusedLocal
def ping(request):
    return HttpResponse('Ok')


def make_locale(raw_locale):
    Locale = namedtuple('Locale', ('language', 'country'))
    lang = country = None
    items = raw_locale.split('_')

    if len(items) >= 1:
        lang = items[0] or None
    if len(items) >= 2:
        country = items[1] or None

    return Locale(language=lang, country=country)


class UpdatesView(HttpClientMixin, APIView):
    # noinspection PyAbstractClass
    class UpdateParamsValidator(serializers.Serializer):
        locale = serializers.CharField(required=True, allow_blank=True)
        device_id = serializers.UUIDField(required=False, default=None)
        market_links = serializers.BooleanField(required=False, default=False)
        type = serializers.MultipleChoiceField(required=False, choices=(
            UpdateTypes.LOCAL,
            UpdateTypes.GOOGLE_PLAY,
            UpdateTypes.FIRMWARE
        ))

        def validate(self, data):
            if (UpdateTypes.GOOGLE_PLAY in data['type'] or data['market_links']) and data['device_id'] is None:
                raise serializers.ValidationError("Valid device_id is required when requesting google play updates")
            return data

    @staticmethod
    def get_response(request_params, tracked_builds, translator):
        # 'default' parameter for MultipleChoiceField doesn't work as expected
        update_types = request_params['type'] or {UpdateTypes.LOCAL}

        result = []

        if UpdateTypes.LOCAL in update_types or UpdateTypes.FIRMWARE in update_types:
            result += get_beta_updates(tracked_builds, translator, selected_types=update_types)

        if UpdateTypes.GOOGLE_PLAY in update_types or request_params['market_links']:
            language = make_locale(request_params['locale']).language
            result += get_market_updates(device_id=request_params['device_id'], language=language)

        return result

    def get(self, request):
        validator = self.UpdateParamsValidator(data=request.query_params)
        validator.is_valid(raise_exception=True)

        request_params = validator.validated_data
        locale = make_locale(request_params['locale'])
        user = create_localization_user(
            uuid=self.uuid,
            locale=locale,
            user_agent=self.user_agent,
            serial_number=self.serial_number,
            yphone_id=self.yphone_id
        )
        config, translator = get_localization_helpers(user)

        tb = TrackedBuilds(config)
        return Response(
            self.get_response(
                request_params,
                translator=translator,
                tracked_builds=get_tracked_builds(
                    config,
                    tb.get_key_prefix(self),
                    self.client_tags
                )
            )
        )


class SettingsView(HttpClientMixin, APIView):
    # noinspection PyAbstractClass
    class LocalizationItemSerializer(serializers.Serializer):
        name = serializers.CharField(required=True)
        value = serializers.CharField(required=True)

    # noinspection PyUnusedLocal
    def get(self, request):
        user = create_localization_user(
            uuid=self.uuid,
            user_agent=self.user_agent,
            serial_number=self.serial_number,
            yphone_id=self.yphone_id
        )

        config, translator = get_localization_helpers(user)
        items = config.get_all_enabled_items()
        serializer = self.LocalizationItemSerializer(
            filter(is_client_property, items),
            many=True
        )
        return Response(serializer.data)


class InternalStorageDownloadView(APIView):
    # noinspection PyAbstractClass
    class ParamsValidator(serializers.Serializer):
        package_name = serializers.CharField(required=True, min_length=1)

    def get(self, request):
        validator = self.ParamsValidator(data=request.query_params)
        validator.is_valid(raise_exception=True)
        package_name = validator.validated_data['package_name']

        try:
            return HttpResponseRedirect(get_sw_url(package_name))
        except AppNotFound:
            raise NotFound('Package name = %s not found in setup wizard promo storage' % package_name)


# noinspection PyAbstractClass
class SubscribeQueryValidator(serializers.Serializer):
    package_name = serializers.CharField(min_length=1, required=True)
    device_id = serializers.UUIDField(required=True)


class BaseSubscribeView(APIView):
    def get_validated_data(self, request):
        validator = SubscribeQueryValidator(data=request.data)
        validator.is_valid(raise_exception=True)
        return validator.validated_data


class UnsubscribeView(BaseSubscribeView):
    def post(self, request):
        data = self.get_validated_data(request)
        package_name = data['package_name']
        device_id = data['device_id']
        num_of_deleted = UpdateSubscriptions.objects(device_id=device_id, package_name=package_name).delete()
        if num_of_deleted > 0:
            return Response()
        raise NotFound()


class SubscribeView(BaseSubscribeView):
    def post(self, request):
        data = self.get_validated_data(request)
        package_name = data['package_name']
        device_id = data['device_id']

        if YandexDistributedApp.objects(package_name=package_name).count() == 0:
            error_response = "package_name=%s not in yandex_distributed_app collection" % package_name
            return Response(error_response, status=status.HTTP_406_NOT_ACCEPTABLE)

        now = datetime.utcnow()
        UpdateSubscriptions.objects(
            device_id=device_id,
            package_name=package_name
        ).update(upsert=True, set__timestamp=now)
        return Response()
