# -*- coding: utf-8 -*-

import logging
import json

import os
import requests
from django.conf import settings

logger = logging.getLogger(__name__)


def geodata_exists(path):
    if os.path.exists(path):
        return True
    logger.error('Geodata file %s does not exist', path)
    return False


if settings.USE_HTTP_GEOBASE or not geodata_exists(settings.GEODATA_BIN_PATH):
    from yaphone.http_geobase_lookuper import httpgeobase

    logger.info('Using HTTP geobase')
    geobase_lookuper = httpgeobase.Lookup(settings.GEOBASE_URL)
else:
    import geobase6

    logger.info('Using local geobase')
    geobase_lookuper = geobase6.Lookup(settings.GEODATA_BIN_PATH)


class LbsLocator(object):
    TIMEOUT = 1

    def __init__(self, api_key):
        self.api_key = api_key

    def locate(self, gsm_cells=None, wifi_networks=None, ip=None, uuid=None):
        """
            gsm_cells=[
               {
                  "countrycode": 250,
                  "operatorid": 99,
                  "cellid": 42332,
                  "lac": 36002,
                  "signal_strength": -80,
                  "age": 5555
               }
           ]

           wifi_networks=[
               {
                  "mac": "00-1C-F0-E4-BB-F5",
                  "signal_strength": -88,
                  "age": 0,
               }
           ]
        """
        data = {
            'common': {
                'version': '1.0',
                'api_key': self.api_key,
            },
        }
        if gsm_cells:
            data['gsm_cells'] = gsm_cells
        if wifi_networks:
            data['wifi_networks'] = wifi_networks
        if ip:
            data['ip'] = {
                'address_v4': ip,
            }
        if uuid:
            params = {'uuid': uuid.hex}
        else:
            params = {}
        try:
            response = requests.post(
                settings.LBS_URL,
                params=params,
                data={'json': json.dumps(data)},
                timeout=self.TIMEOUT
            )
        except Exception as e:
            logger.warning('LBS Locator request error %s', e)
            return None
        if response.ok:
            result = response.json()
            return result['position']  # https://wiki.yandex-team.ru/yandexmobile/lbs/lbsprotocol/geolocation/#primer
        else:
            logger.warning('Request to "%s" return %s: %s', response.url, response.status_code, response.content)
            return None


def get_parent_region_with_type(region_id, parent_region_type):
    for parent_id in geobase_lookuper.get_parents_ids(region_id):
        parent = geobase_lookuper.get_region_by_id(parent_id)
        if parent['type'] == parent_region_type:
            return parent_id


lbs_locator = LbsLocator(settings.LBS_TOKEN)
