from hypothesis import given
import hypothesis.strategies as st
import six

from yaphone.utils.parsed_version import ParsedVersion


def test_init():
    assert ParsedVersion(u'1.2.3').version == [1, 2, 3]
    assert ParsedVersion(u'').version is None
    assert ParsedVersion(u'a.b-c').version == [u'a', u'b', u'c']
    assert ParsedVersion(u'1-a.b-c-2').version == [1, 'a', 'b', 'c', 2]
    assert ParsedVersion(u'1-a').version == [1, 'a']
    assert ParsedVersion(u'1.a').version == [1, 'a']
    assert ParsedVersion(u'do.not.tear.me.apart').version == [u'do', u'not', u'tear', u'me', u'apart']
    assert ParsedVersion(u'123-456').version == [123, 456]
    assert ParsedVersion(u'2.0.9-oemOreo.5002388').version == [2, 0, 9, u'oemOreo', 5002388]


def test_le():
    assert ParsedVersion(u'1.2.3') <= ParsedVersion(u'1.2.4')
    assert not ParsedVersion(u'1.2.3') <= ParsedVersion(u'1.2.2')
    assert ParsedVersion(u'1.5') <= ParsedVersion(u'1.12')
    assert ParsedVersion(u'a.10') <= ParsedVersion(u'1.0')
    assert ParsedVersion(u'a.10') <= ParsedVersion(u'1')
    assert ParsedVersion(u'1.a.b') <= ParsedVersion(u'1.a.c')
    assert ParsedVersion(u'1.az') <= ParsedVersion(u'1.b')
    assert ParsedVersion(u'1.2') <= ParsedVersion(u'1.2.3')
    assert ParsedVersion(u'1.a') <= ParsedVersion(u'1.a.0')
    assert ParsedVersion(u'1.a.0') <= ParsedVersion(u'1.a')
    assert ParsedVersion(u'1.a-2') <= ParsedVersion(u'1-a.2')
    assert ParsedVersion(u'1.2.0') <= ParsedVersion(u'1.2')


@given(
    st.lists(st.one_of(st.integers(min_value=0), st.characters(whitelist_categories=('Lu', 'Ll'))), min_size=1),
    st.lists(st.sampled_from([u'.', u'-'])),
)
def test_init_2(version, delimiters):
    default_delim = u'.'
    version_components = []
    for i, v in enumerate(version):
        version_components.append(v)
        version_components.append(delimiters[i] if i < len(delimiters) else default_delim)
    if version_components:
        version_components.pop()
    assert ParsedVersion(u''.join(map(six.text_type, version_components))).version == version
