import re
from abc import ABC
from typing import Pattern

DEVICE_TV = 'tv'
DEVICE_PHONE = 'phone'
DEVICE_TABLET = 'tablet'
DEFAULT_LANGUAGE = 'ru'

USER_AGENT_STR_PATTERN = (r'^(?P<app_name>[\w.]+)/(?P<app_version_string>[\w.]+)?\s+' +
                          r'\((?P<device_manufacturer>\S+)\s+(?P<device_model>.*);\s+' +
                          r'(?P<os_name>.+)\s+(?P<os_version>.*)\)\s*' +
                          r'(?P<device_type>(?i)({}|{}|{}))?$'.format(
                              DEVICE_PHONE, DEVICE_TABLET, DEVICE_TV
                          ))

USER_AGENT_REGEXP: Pattern[str] = re.compile(USER_AGENT_STR_PATTERN)

# Accept-Language header item: en, en_US, en_us, en-US, *
ACCEPT_LANGUAGE_REGEXP: Pattern[str] = re.compile(
    r'^(?P<language>(\w{2})|\*)([_-](?P<country>[\w]{2}))?'
)

# BRAND/PRODUCT/DEVICE:VERSION.RELEASE/BUILD_ID/BUILD_NUMBER:BUILD_VARIANT/TAGS
BUILD_FINGERPRINT_REGEXP: Pattern[str] = re.compile(
    r'^(?P<brand>[^\s/:]+)/(?P<product>[^\s/:]+)/(?P<device>[^\s/:]+):'
    r'(?P<version_release>[^\s/:]+)/'  # VERSION.RELEASE, e.g.: 7.1.1
    r'(?P<build_id>[^\s/:]+)/(?P<build_number>[^\s/:]+):'
    r'(?P<build_variant>[^\s/:]+)/(?P<tags>[^\s/:]+)$'
)


class RegexpParser(ABC):
    regexp = None

    @classmethod
    def parse(cls, data: str) -> dict:
        match = cls.regexp.match(data)
        if match:
            return match.groupdict()
        else:
            return {}


class UaParser(RegexpParser):
    regexp = USER_AGENT_REGEXP


class AcceptLanguageParser(RegexpParser):
    regexp = ACCEPT_LANGUAGE_REGEXP


class FingerprintParser(RegexpParser):
    regexp = BUILD_FINGERPRINT_REGEXP
