import React, { useState, useCallback } from 'react';

import { Text } from 'src/components/Text';
import { Button } from 'src/components/Button';
import { getPassportURL } from 'src/utils/urlBuilder';
import { authorizeUser } from 'src/dataProvider/authorizeUser';
import { openLink } from 'src/utils/chrome';
import { logInfo, logError } from 'src/logger';

import { AuthorizationButtonType } from './types';

import { setClipboardValue } from 'src/utils/clipboard';

export const AuthorizationButton = props => {
    const { type, account, children, borderType } = props;

    const passportUrl = getPassportURL(account);
    const isLinkButton = type === AuthorizationButtonType.LINK;

    const [isLoading, setIsLoading] = useState(false);

    const onButtonClick = useCallback(async () => {
        if (isLinkButton) {
            logInfo({ message: 'passport-authorization:redirect-to-passport-page' });
            await setClipboardValue(account.password);
            openLink({ href: passportUrl });
            return;
        }

        setIsLoading(true);
        const { isSuccessful, url } = await authorizeUser(account);
        setIsLoading(false);

        if (isSuccessful) {
            logInfo({ message: 'passport-authorization:automation' });
            openLink({ href: url });
        } else {
            // Если не получилось авторизоваться в 1 клик, то отправляем пользователя на страницу паспорта
            logError({ message: 'passport-authorization:fail:automation' });
            await setClipboardValue(account.password);
            openLink({ href: passportUrl });
        }
    }, [account, setIsLoading, isLinkButton, passportUrl]);

    return (
        <Button isLoading={isLoading} onClick={onButtonClick} borderType={borderType}>
            {isLinkButton ? children : <Text withRedFirstLetter>Авторизоваться</Text>}
        </Button>
    );
};

export default AuthorizationButton;
