import React, { useState, useCallback, ChangeEvent } from 'react';
import cn from 'classnames';

import { setClipboardValue } from 'src/utils/clipboard';
import { Text, TextSize, TextColor } from 'src/components/Text';

import { InputSize } from './types';

import styles from './Input.pcss';

const TIMEOUT = 800;

export const Input = props => {
    const {
        className,
        size = InputSize.MEDIUM,
        onChange,
        autoFocus,
        label,
        placeholder,
        initialValue = '',
        readOnly,
        withoutBorder = false,
        maxLength,
        onFocus = () => {},
    } = props;

    const [value, setValue] = useState(initialValue);
    const [isFocused, setIsFocused] = useState(autoFocus);
    const [info] = useState('Скопировано');
    const [isVisibleInfo, setIsVisibleInfo] = useState(false);

    const isValue = Boolean(value);
    const isUpLabel = isValue || isFocused;

    const onChangeCallback = useCallback(
        event => {
            const {
                currentTarget: { value },
            } = event;

            setValue(value);
            onChange(event);
        },
        [value, setValue, onChange]
    );

    const onClickCallback = useCallback(
        async event => {
            if (!readOnly && !isVisibleInfo) {
                return;
            }

            const {
                currentTarget: { value },
            } = event;

            await setClipboardValue(value);
            setIsVisibleInfo(true);
            setTimeout(() => {
                setIsVisibleInfo(false);
            }, TIMEOUT);
        },
        [readOnly, setIsVisibleInfo, isVisibleInfo]
    );

    const onFocusCallback = useCallback(() => {
        onFocus();
        setIsFocused(true);
    }, [setIsFocused]);

    const onBlurCallback = useCallback(() => {
        setIsFocused(false);
    }, [setIsFocused]);

    const rootClassNames = cn(
        styles.root,
        {
            [styles.root_withValue]: isValue,
            [styles.root_focused]: isFocused,
            [styles.root_readOnly]: readOnly,
            [styles.root_withoutBorder]: withoutBorder,
            [styles.root_sizeSmall]: size === InputSize.SMALL,
            [styles.root_withoutLabel]: label === undefined,
        },
        className
    );

    const labelTextProps = {
        withRedFirstLetter: isUpLabel,
        size: isUpLabel ? TextSize.TINY : TextSize.MEDIUM,
        color: isUpLabel ? undefined : TextColor.PLACEHOLDER,
    };

    return (
        <div className={rootClassNames}>
            <div className={cn(styles.info, { [styles.info_visible]: isVisibleInfo })}>
                <Text withRedFirstLetter>{info}</Text>
            </div>
            {label && (
                <label className={styles.label}>
                    <Text {...labelTextProps}>{label}</Text>
                </label>
            )}
            <input
                type="text"
                placeholder={placeholder ? placeholder : null}
                onClick={onClickCallback}
                autoFocus={autoFocus}
                defaultValue={initialValue}
                onChange={onChangeCallback}
                className={styles.input}
                onFocus={onFocusCallback}
                onBlur={onBlurCallback}
                readOnly={readOnly}
                maxLength={maxLength}
            />
        </div>
    );
};

export default Input;
