import React, { useCallback, useState } from 'react';
import cn from 'classnames';

import { Button, Text, ButtonType, NewVersion, Link, LinkType, LinkStyle } from 'src/components';
import { MenuIcon } from 'src/icons';
import { CurrentVersion } from 'src/components/CurrentVersion';
import { URL } from 'src/constants/urls';
import { Path } from 'src/types/navigation';
import { trackerTask } from 'src/entities/TrackerTask';

import styles from './LeftMenu.pcss';

const TIMEOUT_DURATION = 500;

export const LeftMenu = props => {
    const { actualVersion, setActualVersion } = props;

    const [isVisible, setIsVisible] = useState(false);
    const [canShow, setCanShow] = useState(true);

    const onMouseEnter = useCallback(() => {
        if (canShow) {
            setIsVisible(true);
        }
    }, [canShow, setIsVisible]);

    const onMouseLeave = useCallback(() => {
        setIsVisible(false);
    }, [setIsVisible]);

    const hideMenu = useCallback(() => {
        setIsVisible(false);
        setCanShow(false);
        setTimeout(() => setCanShow(true), TIMEOUT_DURATION);
    }, [setIsVisible, setCanShow]);

    const onNewOAuthTokenButtonClick = useCallback(() => {
        hideMenu();
    }, [hideMenu]);

    const onShowHistoryButtonClick = useCallback(() => {
        hideMenu();
    }, [hideMenu]);

    const newVersionProps = {
        actualVersion,
        setActualVersion,
    };

    const currentVersionProps = {
        actualVersion,
    };

    return (
        <div
            onMouseEnter={onMouseEnter}
            onMouseLeave={onMouseLeave}
            className={cn(styles.root, isVisible ? styles.visible : styles.hidden)}
        >
            <div className={styles.rightColumn}>
                <MenuIcon className={styles.captionIcon} />
                <CurrentVersion {...currentVersionProps} />
            </div>
            <div className={styles.captionWrapper}>
                <Text withRedFirstLetter>Меню</Text>
            </div>
            <Link type={LinkType.ROUTER} style={LinkStyle.fullWidth} href={Path.Accounts}>
                <Button type={ButtonType.LEFT_MENU} onClick={onShowHistoryButtonClick}>
                    <Text withRedFirstLetter>Аккаунты</Text>
                </Button>
            </Link>
            <Link type={LinkType.ROUTER} style={LinkStyle.fullWidth} href={Path.OAuth}>
                <Button type={ButtonType.LEFT_MENU} onClick={onNewOAuthTokenButtonClick}>
                    <Text withRedFirstLetter>OAuth токен</Text>
                </Button>
            </Link>
            <Link type={LinkType.LEFT_MENU} href={URL.FEEDBACK_FORM}>
                <Text withRedFirstLetter>Оставить feedback</Text>
            </Link>
            <Link type={LinkType.LEFT_MENU} href={URL.USER_TRACKER_TASK(trackerTask.get())}>
                <Text withRedFirstLetter>Персональные данные</Text>
            </Link>
            <NewVersion {...newVersionProps} />
        </div>
    );
};

export default LeftMenu;
