import { URL } from 'src/constants/urls';
import { Header, RequestMethod } from 'src/dataProvider/types';

import { oAuthToken } from 'src/entities/OAuthToken';
import { trackerTask } from 'src/entities/TrackerTask';
import { SYNCHRONIZATION_DATA_MARKER } from 'src/entities/SynchronizationData/types';
import { logError, logInfo } from 'src/logger';

export async function fetchSynchronizationData() {
    const requestURL = `${URL.TRACKER}/issues/${trackerTask.get()}/comments?perPage=1`;
    logInfo({ message: 'data-provider:fetchSynchronizationData:start' });

    const headers = {
        [Header.AUTHORIZATION]: `OAuth ${oAuthToken.get()}`,
    };

    const response = await fetch(requestURL, { headers, method: RequestMethod.GET });

    const responseText = await response.text();

    try {
        // Изначально считаем, что комментарий может существовать
        // Если commentCanExist === true, то новый комментарий создавать нельзя!
        let synchronizationDataCanExist = true;
        // Содержание комментария
        let synchronizationData = '';
        // ID комментарий
        let synchronizationDataId = undefined;

        const responseData = JSON.parse(responseText);

        // Если ручка успешно отработала и комментарий не нашелся
        if (Array.isArray(responseData) && !responseData.length) {
            // То считаем, что комментария точно нет и можно попытаться создать его для пользователя
            synchronizationDataCanExist = false;
        } else if (Array.isArray(responseData) && responseData.length === 1) {
            // Запрос можно считать действительно успешным если ручка вернула только один комментарий
            synchronizationData = responseData[0].text;
            synchronizationDataId = responseData[0].id;
        }

        // Удалось ли найти коммментарий с данными
        const isSuccess = synchronizationData.includes(SYNCHRONIZATION_DATA_MARKER);

        if (isSuccess) {
            logInfo({ message: 'data-provider:fetchSynchronizationData:success' });
        } else {
            logError({ message: 'data-provider:fetchSynchronizationData:fail' });
        }

        return {
            synchronizationDataCanExist,
            isSuccess,
            synchronizationData,
            synchronizationDataId,
        };
    } catch (error) {
        console.error('fetchSynchronizationData response: ', responseText);
        console.error('fetchSynchronizationData error: ', error);

        logError({ message: 'data-provider:fetchSynchronizationData:fail' });

        return {
            synchronizationDataCanExist: true,
            isSuccess: false,
            synchronizationData: '',
            synchronizationDataId: undefined,
        };
    }
}
