import { Account } from 'src/entities/Account';

import { getAccountsAsList } from './helpers';
import { getSavedAccounts, saveAccounts, clearSavedAccounts } from './localStorage';
import { syncAccounts } from './methods/syncAccounts';
import { exportAccounts } from './methods/export';
import { importAccounts } from './methods/import';

class AccountsManager {
    constructor() {
        const savedAccounts = getSavedAccounts();
        this.syncAccounts = syncAccounts.bind(this);
        this.exportAccounts = exportAccounts.bind(this);
        this.importAccounts = importAccounts.bind(this);

        this._accounts = savedAccounts.reduce((result, account) => {
            const { uid } = account;

            if (!uid) {
                // Если нет uid, то пропускаем аккаунт
                return result;
            }

            result[account.uid] = new Account(account);
            return result;
        }, {});
    }

    _get() {
        return this._accounts;
    }

    // Можно использовать и для обновления путем замены
    add(accountData) {
        const { uid } = accountData;

        if (!uid) {
            return;
        }

        this._accounts[accountData.uid] = new Account(accountData);
        this._save();
    }

    async toggleBookmark(uid) {
        const account = this._accounts[uid];

        await account.toggleBookmark();
        this._save();

        return account.get();
    }

    removeAccount(uid) {
        delete this._accounts[uid];
        this._save();
    }

    clear() {
        this._accounts = {};
        clearSavedAccounts();
    }

    getByUid(uid) {
        return this._accounts[uid];
    }

    /**
     * @param {function[]} filters
     * filter - функция, которая возвращает отфильтрованный список аккаунтов вида {[key: uid]: Account}
     */
    getFiltered(filters) {
        return filters.reduce((result, element) => {
            const { filter, params = {} } = element;

            if (!filter) {
                return result;
            }

            // Фильтр, созданный с помощью функции createFilter
            return filter(result, params);
        }, this._get());
    }

    /** Аналогична getFiltered, но возвращает аккаунты в виде Account[] */
    getFilteredList(filters) {
        const filteredAccounts = this.getFiltered(filters);

        return getAccountsAsList(filteredAccounts);
    }

    /** Private methods */
    _save() {
        saveAccounts(getAccountsAsList(this._accounts));
    }
}

export const accountsManager = new AccountsManager();
