import React, { useCallback, useEffect, useState } from 'react';

import {
    Button,
    Text,
    Input,
    InputSize,
    ButtonType,
    TextTheme,
    Link,
    LinkType,
} from 'src/components';
import { Account } from 'src/components/Account';
import { Spinner, Size } from 'src/components/Spinner';
import { accountsManager } from 'src/entities/AccountsManager';
import { someFieldsByText } from 'src/entities/AccountsManager/filters/someFieldsByText';
import { accountUsefulFields } from 'src/entities/Account/types';
import { environmentFilters, getFilterIdByName } from 'src/entities/AccountsManager/filters/types';
import { Path } from 'src/types/navigation';
import { usePageAnalytics } from 'src/utils/yandexMetrika';
import { logError, logInfo } from 'src/logger';

import styles from './Accounts.pcss';

export const Accounts = () => {
    const [accounts, setAccounts] = useState([]);
    const [searchInputValue, setInputFilter] = useState('');
    const [clearButtonState, setClearButtonState] = useState({ click: 0 });
    const [environmentFilterIndex, setEnvironmentFilterIndex] = useState(0);
    const [isLoading, setIsLoading] = useState(false);

    usePageAnalytics('Accounts');

    const { name: environmentFilterName, filter: environmentFilter } = environmentFilters[
        environmentFilterIndex
    ];

    const filters = [
        { filter: environmentFilter },
        {
            filter: someFieldsByText,
            params: {
                fields: accountUsefulFields,
                text: searchInputValue,
            },
        },
    ];

    useEffect(() => {
        setAccounts(accountsManager.getFilteredList(filters));
    }, [clearButtonState, environmentFilterIndex, searchInputValue]);

    const isAccounts = accounts.length !== 0;

    const onClearButtonClick = useCallback(() => {
        const { click } = clearButtonState;

        accountsManager.clear();
        setClearButtonState({ click: click + 1 });

        logInfo({ message: 'accounts-list:clear-data-button:click' });
    }, [clearButtonState, setClearButtonState]);

    const onFilterButtonClick = useCallback(() => {
        let newEnvironmentFilterIndex = environmentFilterIndex + 1;

        if (newEnvironmentFilterIndex >= environmentFilters.length) {
            newEnvironmentFilterIndex = 0;
        }

        setEnvironmentFilterIndex(newEnvironmentFilterIndex);

        logInfo({ message: 'accounts-list:filter-button:click' });
    }, [environmentFilterIndex, setEnvironmentFilterIndex]);

    const handleInputFilterChange = useCallback(event => setInputFilter(event.target.value), [
        setInputFilter,
    ]);

    const onExportButtonClick = useCallback(() => {
        accountsManager.exportAccounts();

        logInfo({ message: 'accounts-list:export-button:click' });
    }, []);

    const onSyncButtonClick = useCallback(async () => {
        setIsLoading(true);
        await accountsManager.syncAccounts();
        setEnvironmentFilterIndex(getFilterIdByName('Bookmarks'));
        setAccounts(accountsManager.getFilteredList(filters));
        setIsLoading(false);

        logInfo({ message: 'accounts-list:sync-button:click' });
    }, [filters]);

    const onImportInputChangeClick = useCallback(async event => {
        const fileData = await event.currentTarget.files[0].text();
        const accounts = JSON.parse(fileData);
        accountsManager.importAccounts(accounts);
        setAccounts([]);
        setAccounts(accountsManager.getFilteredList(filters));

        logInfo({ message: 'accounts-list:import-button:click' });
    }, []);

    const onImportInputClear = useCallback(async event => {
        event.currentTarget.value = '';
    }, []);

    const accountsList = isAccounts ? (
        accounts
            .sort((first, second) => second.date - first.date)
            .map(account => <Account account={account} key={account.login} />)
    ) : (
        <Text className={styles.noAccountsMessage} withRedFirstLetter>
            Сохраненных аккаунтов нет
        </Text>
    );

    const spinner = (
        <div className={styles.spinnerWrapper}>
            <Spinner size={Size.IN_BUTTON} />
        </div>
    );

    return (
        <div className={styles.root}>
            <div className={styles.menu}>
                <Text className={styles.title} withRedFirstLetter>
                    Мои аккаунты
                </Text>

                <Input
                    className={styles.filterInput}
                    size={InputSize.SMALL}
                    placeholder="Поиск ..."
                    onChange={handleInputFilterChange}
                />

                <Text className={styles.title}>Окружение:</Text>
                <Button
                    className={styles.button}
                    type={ButtonType.FIT_CONTENT}
                    onClick={onFilterButtonClick}
                >
                    <Text withRedFirstLetter>{environmentFilterName}</Text>
                </Button>

                <Link type={LinkType.ROUTER} href={Path.Root}>
                    <Button className={styles.button} type={ButtonType.FIT_CONTENT}>
                        <Text withRedFirstLetter>Закрыть</Text>
                    </Button>
                </Link>
            </div>

            <div className={styles.accountsList}>{isLoading ? spinner : accountsList}</div>

            <div className={styles.footer}>
                <div className={styles.footer__actions}>
                    <Text withRedFirstLetter theme={TextTheme.LINK} onClick={onSyncButtonClick}>
                        Синхронизировать
                    </Text>
                    <Text>
                        {'  '}-{'  '}
                    </Text>

                    <div className={styles.importElementsWrapper}>
                        <Text withRedFirstLetter theme={TextTheme.LINK}>
                            Импортировать
                        </Text>
                        <label className={styles.importLabel}>
                            <input
                                className={styles.fileInput}
                                type="file"
                                onInput={onImportInputChangeClick}
                                onClick={onImportInputClear}
                            />
                        </label>
                    </div>

                    <Text>
                        {'  '}-{'  '}
                    </Text>
                    <Text withRedFirstLetter theme={TextTheme.LINK} onClick={onExportButtonClick}>
                        Экспортировать
                    </Text>
                    <Text>
                        {'  '}-{'  '}
                    </Text>
                    <Text withRedFirstLetter theme={TextTheme.LINK} onClick={onClearButtonClick}>
                        Очистить список
                    </Text>
                </div>
            </div>
        </div>
    );
};
