import React, { useCallback, useState } from 'react';

import { Environment } from 'src/constants/environment';
import { getCurrentDate } from 'src/utils/date';
import { getNewAccount } from 'src/dataProvider/getNewAccount';
import { Button, Text, Redirects, Settings } from 'src/components';
import { Account } from 'src/components/Account';
import { yandexTeamLogin } from 'src/entities/YandexTeamLogin';
import { accountLogin } from 'src/entities/AccountLogin';
import { accountPassword } from 'src/entities/AccountPassword';
import { accountsManager } from 'src/entities/AccountsManager';
import { sendCommonMetrics, usePageAnalytics } from 'src/utils/yandexMetrika';
import { logInfo, logError, updateAdditional } from 'src/logger';

import styles from './Main.pcss';

export const Main = () => {
    const [account, setAccount] = useState(undefined);
    const [isLoadingTest, setIsLoadingTest] = useState(false);
    const [isLoadingProd, setIsLoadingProd] = useState(false);

    usePageAnalytics('Main');

    const onSuccessCallback = useCallback(
        environment => xhr => {
            const result = JSON.parse(xhr.target.response);
            const { account: newAccount } = result;

            const modifiedAccount = {
                ...newAccount,
                environment,
                date: getCurrentDate(),
            };

            setAccount(modifiedAccount);
            sendCommonMetrics({ generatedAccount: 'Аккаунт создан и отображен' });
            accountsManager.add(modifiedAccount);

            setIsLoadingTest(false);
            setIsLoadingProd(false);

            updateAdditional({
                account: { uid: modifiedAccount.uid, login: modifiedAccount.login },
            });
            logInfo({ message: 'accountCreating:success' });
            updateAdditional({
                account: undefined,
            });
        },
        [setAccount]
    );

    const onErrorCallback = useCallback(() => {
        logError({ message: 'accountCreating:fail' });
        sendCommonMetrics({ generatedAccount: 'Не удалось сгенерировать аккаунт' });

        setIsLoadingTest(false);
        setIsLoadingProd(false);
    }, []);

    const onTestingButtonClick = useCallback(() => {
        logInfo({ message: 'accountCreating:testing-try' });
        sendCommonMetrics({ generatedAccount: 'Попытка сгенерировать тестовый аккаунт' });

        setAccount(undefined);
        setIsLoadingTest(true);
        getNewAccount({
            onErrorCallback,
            onSuccessCallback: onSuccessCallback(Environment.TESTING),
            params: {
                yandexTeamLogin: yandexTeamLogin.get(),
                customLogin: accountLogin.get(),
                password: accountPassword.get(),
                environment: Environment.TESTING,
            },
        });
    }, [
        onSuccessCallback,
        setIsLoadingTest,
        yandexTeamLogin.get,
        accountLogin.get,
        accountPassword.get,
    ]);

    const onProductionButtonClick = useCallback(() => {
        logInfo({ message: 'accountCreating:production-try' });
        sendCommonMetrics({ generatedAccount: 'Попытка сгенерировать продовый аккаунт' });

        setAccount(undefined);
        setIsLoadingProd(true);
        getNewAccount({
            onErrorCallback,
            onSuccessCallback: onSuccessCallback(Environment.PRODUCTION),
            params: {
                yandexTeamLogin: yandexTeamLogin.get(),
                customLogin: accountLogin.get(),
                password: accountPassword.get(),
                environment: Environment.PRODUCTION,
            },
        });
    }, [
        onSuccessCallback,
        setIsLoadingProd,
        yandexTeamLogin.get,
        accountLogin.get,
        accountPassword.get,
    ]);

    return (
        <div className={styles.root}>
            <Redirects />
            <Settings />
            <Button
                onClick={onTestingButtonClick}
                className={styles.button}
                isLoading={isLoadingTest}
                autoFocus
            >
                <Text withRedFirstLetter>Testing</Text>
            </Button>
            <Button
                onClick={onProductionButtonClick}
                className={styles.button}
                isLoading={isLoadingProd}
            >
                <Text withRedFirstLetter>Production</Text>
            </Button>
            {account && (
                <div className={styles.successInfo}>
                    <Account account={account} />
                </div>
            )}
        </div>
    );
};
