import React, { useState, useCallback } from 'react';

import { Link, Text, Input, Button, LinkType, LinkStyle } from 'src/components';
import { URL } from 'src/constants/urls';
import { setClipboardValue } from 'src/utils/clipboard';
import { oAuthToken } from 'src/entities/OAuthToken';
import { Path } from 'src/types/navigation';
import { Redirect } from 'react-router';
import { usePageAnalytics } from 'src/utils/yandexMetrika';

import { useOnSubmitButtonClick } from './hooks/useOnSubmitButtonClick';

import styles from './OAuth.pcss';

export const OAuth = () => {
    const [inputValue, setInputValue] = useState(oAuthToken.get());
    const [isRootRedirect, setIsRootRedirect] = useState(false);

    usePageAnalytics('OAuth');

    const onInputChange = useCallback(
        event => {
            setInputValue(event.target.value);
        },
        [setInputValue]
    );

    const onSubmitButtonClick = useOnSubmitButtonClick({
        inputValue,
        setIsRootRedirect,
    });

    const onCopyButtonClick = useCallback(() => {
        setClipboardValue(oAuthToken.get());
    }, []);

    const submitButtonProps = {
        className: styles.button,
        onClick: onSubmitButtonClick,
    };

    const copyButtonProps = {
        ...submitButtonProps,
        onClick: onCopyButtonClick,
    };

    const inputProps = {
        autoFocus: true,
        initialValue: oAuthToken.get(),
        onChange: onInputChange,
        label: 'OAuth токен',
    };

    if (isRootRedirect) {
        return <Redirect to={Path.Root} />;
    }

    return (
        <div className={styles.root}>
            <div className={styles.info}>
                <Text>Для использования расширения необходимо указать свой OAuth токен</Text>
                <div className={styles.link}>
                    <Link href={URL.OAUTH}>
                        <Text withRedFirstLetter>Получить OAuth токен</Text>
                    </Link>
                </div>
            </div>
            <div className={styles.form}>
                <div className={styles.input}>
                    <Input {...inputProps} />
                </div>

                <Button {...submitButtonProps}>
                    <Text withRedFirstLetter>Использовать</Text>
                </Button>

                {oAuthToken.get() && (
                    <Link type={LinkType.ROUTER} style={LinkStyle.fullWidth} href={Path.Root}>
                        <Button {...copyButtonProps}>
                            <Text withRedFirstLetter>Копировать</Text>
                        </Button>
                    </Link>
                )}
            </div>
        </div>
    );
};

export default OAuth;
