import { useCallback } from 'react';
import { oAuthToken } from 'src/entities/OAuthToken';

import { userConsumer } from 'src/entities/UserConsumer';
import { yandexTeamLogin } from 'src/entities/YandexTeamLogin';
import { trackerTask } from 'src/entities/TrackerTask';
import { synchronizationData as synchronizationDataEntity } from 'src/entities/SynchronizationData';
import { modal } from 'src/entities/Modal';
import { Errors } from 'src/types/errors';
import { sendCommonMetrics } from 'src/utils/yandexMetrika';
import { logInfo, logError, updateAdditional } from 'src/logger';

async function tryAuthUser(params) {
    const { inputValue, setIsRootRedirect } = params;
    logInfo({ message: 'tryAuthUser:start' });

    // Считаем oAuth токен правильным и используем его
    oAuthToken.set(inputValue);

    // Сначала пытаемся получить staff-логин пользователя
    const login = await yandexTeamLogin.tryFetchYandexTeamLogin();

    if (!login) {
        oAuthToken.clear();

        // Показываем сообщение об ошибке
        modal.setData({
            type: 'error',
            message: Errors.CAN_NOT_GET_STAFF_LOGIN,
            retryButtonText: 'Попробовать еще раз',
            callback: () => tryAuthUser(params),
        });
        modal.show();
        logError({ message: 'tryAuthUser:fail:can-not-get-staff-login' });

        return;
    }

    updateAdditional({ user: login });
    logInfo({ message: 'tryAuthUser:success:get-login' });
    sendCommonMetrics({ tryAuthUser_login: login });

    // Потом пытаемся создать/проверить userConsumer пользователя
    const consumer = await userConsumer.tryCreateConsumer();

    if (!consumer) {
        oAuthToken.clear();

        // Показываем сообщение об ошибке
        modal.setData({
            type: 'error',
            message: Errors.CAN_NOT_GET_USER_CONSUMER,
            retryButtonText: 'Попробовать еще раз',
            callback: () => tryAuthUser(params),
        });
        modal.show();
        logError({ message: 'tryAuthUser:fail:can-not-get-user-consumer' });

        return;
    }

    updateAdditional({ userConsumer: consumer });
    logInfo({ message: 'tryAuthUser:success:get-user-consumer' });
    sendCommonMetrics({ tryAuthUser_consumer: consumer });

    // Дальше получаем/создаем тикет в трекере для хранения данных пользователя
    const { taskKey } = await trackerTask.tryFetchTrackerTask();

    if (!taskKey) {
        oAuthToken.clear();

        // Показываем сообщение об ошибке
        modal.setData({
            type: 'error',
            message: Errors.CAN_NOT_GET_TASK_NUMBER,
            retryButtonText: 'Попробовать еще раз',
            callback: () => tryAuthUser(params),
        });
        modal.show();
        logError({ message: 'tryAuthUser:fail:can-not-get-task-number' });

        return;
    }

    logInfo({ message: 'tryAuthUser:success:get-task-number' });
    updateAdditional({ taskKey: taskKey });
    sendCommonMetrics({ tryAuthUser_taskKey: taskKey });

    // Дальше получаем/создаем комментарий в тикете для хранения данных пользователя
    const { synchronizationData } = await synchronizationDataEntity.tryFetchSynchronizationData();

    if (!synchronizationData) {
        oAuthToken.clear();

        // Показываем сообщение об ошибке
        modal.setData({
            type: 'error',
            message: Errors.CAN_NOT_GET_TASK_NUMBER,
            retryButtonText: 'Попробовать еще раз',
            callback: () => tryAuthUser(params),
        });
        modal.show();
        logError({ message: 'tryAuthUser:fail:can-not-get-user-data-from-st-task' });

        return;
    }

    logInfo({ message: 'tryAuthUser:success:user-was-authorized' });
    updateAdditional({ taskKey: taskKey });
    sendCommonMetrics({ tryAuthUser_successAuth: true });
    setIsRootRedirect(true);
}

export function useOnSubmitButtonClick(params) {
    const { inputValue, setIsRootRedirect } = params;

    return useCallback(
        async () =>
            tryAuthUser({
                inputValue,
                setIsRootRedirect,
            }),
        [inputValue, setIsRootRedirect]
    );
}
